test_that("nextsample updates fields correctly without compute_lower", {
  # Setup initial object with dummy sample_G returning c(1,0,1)
  R <- list(
    epsilon = 0.05,
    sample_G = function() c(1,0,1),
    S = 5,
    n = 10,
    ptilde = 0.4,
    Ltilde = 0.2,
    B = 5,
    Bn = 10
  )
  class(R) <- "avseqmc_progress"

  R_new <- nextsample(R, compute_lower = FALSE)

  expect_s3_class(R_new, "avseqmc_progress")

  # B and Bn appended with sum and length of sample_G output
  expect_equal(tail(R_new$B, 1), sum(c(1,0,1)))
  expect_equal(tail(R_new$Bn, 1), length(c(1,0,1)))
  expect_equal(R_new$S, 5 + sum(c(1,0,1)))
  expect_equal(R_new$n, 10 + length(c(1,0,1)))

  # ptilde updated with cummin of previous and new
  expect_equal(R_new$ptilde[length(R_new$ptilde)], min(0.4, Un_tilde(R_new$S, R_new$n, R_new$epsilon) + R_new$epsilon))

  # Ltilde appended with NA values equal to length of new batch
  expect_equal(tail(R_new$Ltilde, length(c(1,0,1))), rep(NA_real_, length(c(1,0,1))))
})

test_that("nextsample updates fields correctly with compute_lower = TRUE", {
  R <- list(
    epsilon = 0.05,
    sample_G = function() c(0,1),
    S = 2,
    n = 5,
    ptilde = 0.3,
    Ltilde = c(0.1, 0.15),
    B = c(1, 1),
    Bn = c(3, 2)
  )
  class(R) <- "avseqmc_progress"

  R_new <- nextsample(R, compute_lower = TRUE)

  expect_s3_class(R_new, "avseqmc_progress")

  # B and Bn appended
  expect_equal(tail(R_new$B, 1), sum(c(0,1)))
  expect_equal(tail(R_new$Bn, 1), length(c(0,1)))

  # S and n updated
  expect_equal(R_new$S, 2 + sum(c(0,1)))
  expect_equal(R_new$n, 5 + length(c(0,1)))

  # ptilde updated with cummin of previous and new
  expect_equal(R_new$ptilde[length(R_new$ptilde)], min(tail(R$ptilde, 1), Un_tilde(R_new$S, R_new$n, R_new$epsilon) + R_new$epsilon))

  # Ltilde appended with max of previous Ltilde and new Ln_tilde
  expected_max <- max(c(tail(R$Ltilde, 1), Ln_tilde(R_new$S, R_new$n, R_new$epsilon)), na.rm = TRUE)
  expect_equal(tail(R_new$Ltilde, 1), expected_max)
})

test_that("nextsample handles empty B and Bn vectors", {
  R <- list(
    epsilon = 0.05,
    sample_G = function() c(1),
    S = 0,
    n = 0,
    ptilde = 1,
    Ltilde = numeric(0),
    B = numeric(0),
    Bn = numeric(0)
  )
  class(R) <- "avseqmc_progress"

  R_new <- nextsample(R, compute_lower = TRUE)
  expect_equal(R_new$B, 1)
  expect_equal(R_new$Bn, 1)
  expect_equal(R_new$S, 1)
  expect_equal(R_new$n, 1)
  expect_equal(length(R_new$ptilde), 2)
  expect_equal(length(R_new$Ltilde), 1 + length(numeric(0))) # was 0 + 1 appended
})

test_that("nextsample preserves previous values and adds new batch", {
  R <- list(
    epsilon = 0.01,
    sample_G = function() c(0,1,1),
    S = 3,
    n = 10,
    ptilde = c(0.5, 0.4),
    Ltilde = c(0.3, 0.35),
    B = c(1, 2),
    Bn = c(5, 5)
  )
  class(R) <- "avseqmc_progress"

  R_new <- nextsample(R, compute_lower = FALSE)

  expect_equal(length(R_new$B), length(R$B) + 1)
  expect_equal(length(R_new$Bn), length(R$Bn) + 1)
  expect_equal(length(R_new$ptilde), length(R$ptilde) + 1)
  expect_equal(length(R_new$Ltilde), length(R$Ltilde) + length(c(0,1,1)))
})
