% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bcvusMAR.R
\name{vus_mar}
\alias{vus_mar}
\title{Estimation methods for volume under ROC surface (VUS) under MAR}
\usage{
vus_mar(
  method = "full",
  diag_test,
  dise_vec,
  veri_stat,
  rho_est = NULL,
  pi_est = NULL,
  ci = TRUE,
  ci_level = ifelse(ci, 0.95, NULL),
  boot = FALSE,
  n_boot = ifelse(ci, 250, NULL),
  parallel = FALSE,
  ncpus = ifelse(parallel, detectCores()/2, NULL),
  trace = TRUE
)
}
\arguments{
\item{method}{name of bias-corrected estimation method to be used for estimating the VUS in presence of verification bias. See \code{\link{rocs}} for more details.}

\item{diag_test}{a numeric vector containing the diagnostic test values. \code{NA} values are not admitted.}

\item{dise_vec}{a n * 3  binary matrix with the three columns, corresponding to three classes of the disease status. In row i, 1 in column j indicates that the i-th subject belongs to class j, with j = 1, 2, 3. A row of \code{NA} values indicates a non-verified subject.}

\item{veri_stat}{a binary vector containing the verification status (1 verified, 0 not verified).}

\item{rho_est}{a result of a call to \code{\link{rho_mlogit}} of \code{\link{rho_knn}} to fit the disease model.}

\item{pi_est}{a result of a call to \code{\link{psglm}} to fit the verification model.}

\item{ci}{a logical value. If TRUE (default), computes an confidence interval of VUS and tests the null hypothesis H0: VUS = 1/6.}

\item{ci_level}{an confidence level to be used for constructing the confidence interval; default 0.95.}

\item{boot}{a logical value. Default = \code{FALSE}. If set to \code{TRUE}, a bootstrap resampling is employed to estimate the asymptotic variance of bias-corrected VUS estimates. See \code{\link{asy_var_vus}}.}

\item{n_boot}{the number of bootstrap replicates, which is used for FULL or KNN estimator, or option \code{boot = TRUE}. Usually this will be a single positive integer.}

\item{parallel}{a logical value. If \code{TRUE}, a parallel computing is employed to the bootstrap resampling process.}

\item{ncpus}{number of processes to be used in parallel computing. Default is a half of available cores.}

\item{trace}{a logical value. If \code{TRUE}, tracing information on the progress of the estimation is produced.}
}
\value{
\code{vus_mar} returns an object of class inheriting from "vus_mar" class.

The function \code{\link{print.vus_mar}} can be used to print a summary of the results.

An object of class "vus_mar" is a list containing at least the following components:

\item{vus_fit}{the estimate of VUS.}
\item{std}{the standard error, obtained by using asymptotic theory or bootstrap resampling method.}
\item{call}{the matched call.}
\item{t_stat}{t-statistic.}
\item{p_val_norm}{p-value correspond to normal-test.}
\item{ci_norm}{the confidence interval of VUS by using normal approximation.}
\item{ci_logit}{the confidence interval of VUS via logit transform.}
\item{ci_level}{the confidence level used.}
\item{boot}{the value of \code{boot}.}
\item{n_boot}{the number of bootstrap replicates used.}

In addition, the name of method used to estimate VUS also is given as the attribute of \code{vus_fit}.
}
\description{
\code{vus_mar} computes bias-corrected estimates of the volume under the ROC surface for evaluating the accuracy of a continuous diagnostic test.
}
\details{
The function implements five bias-corrected estimation methods in To Duc et al. (2016, 2020) for estimating VUS of a three-class continuous diagnostic test in presence of verification bias. The estimators are full imputation (FI), mean score imputation (MSI), inverse probability weighted (IPW), semiparametric efficient (SPE) and K nearest-neighbor (KNN), see \code{\link{rocs}}. These estimators work under MAR assumption.

The standard error of the estimates are obtained through the function \code{\link{asy_var_vus}}. In particular, the standard error of the FULL estimate is computed by bootstrap resampling method or by Jackknife approach proposed in Guangming et al. (2013). For the bias-corrected estimates, the standard errors are computed by using asymptotic theory (with respect to FI, MSI, IPW and SPE estimator) or bootstrap resampling method (with respect to KNN estimator). A confidence interval for VUS also is given. A logit transformation is also applied for obtaining the confidence interval.

The default value of the number of bootstrap replicates is 250.

Note that, before apply the functions \code{vus_mar}, the use of \code{\link{pre_data}} might be needed to check the monotone ordering disease classes and to create the matrix format for disease status.
}
\examples{
data(EOC)
head(EOC)


\dontrun{
# FULL data estimator
dise_full <- pre_data(EOC$D.full, EOC$CA125)
dise_vec_full <- dise_full$dise_vec
vus_mar("full", diag_test = EOC$CA125, dise_vec = dise_vec_full)
}

\dontrun{
# Preparing the missing disease status
dise_na <- pre_data(EOC$D, EOC$CA125)
dise_vec_na <- dise_na$dise_vec
dise_fact_na <- dise_na$dise
# FI estimator
rho_out <- rho_mlogit(dise_fact_na ~ CA125 + CA153 + Age, data = EOC,
                      test = TRUE)

vus_mar("fi", diag_test = EOC$CA125, dise_vec = dise_vec_na,
        veri_stat = EOC$V, rho_est = rho_out)

# MSI estimator
vus_mar("msi", diag_test = EOC$CA125, dise_vec = dise_vec_na,
        veri_stat = EOC$V, rho_est = rho_out)

# IPW estimator
pi_out <- psglm(V ~ CA125 + CA153 + Age, data = EOC, test = TRUE)
vus_mar("ipw", diag_test = EOC$CA125, dise_vec = dise_vec_na,
        veri_stat = EOC$V, pi_est = pi_out)

# SPE estimator
vus_mar("spe", diag_test = EOC$CA125, dise_vec = dise_vec_na,
        veri_stat = EOC$V, rho_est = rho_out, pi_est = pi_out)

# KNN estimator, K = 1, Mahalanobis distance
x_mat <- cbind(EOC$CA125, EOC$CA153, EOC$Age)
rho_maha_1nn <- rho_knn(x_mat = x_mat, dise_vec = dise_vec_na,
                        veri_stat = EOC$V, k = 1, type = "mahala")
vus_mar("knn", diag_test = EOC$CA125, dise_vec = dise_vec_na,
        veri_stat = EOC$V, rho_est = rho_maha_1nn)
}

}
\references{
To Duc, K., Chiogna, M. and Adimari, G. (2020)
Nonparametric estimation of ROC surfaces in presence of verification bias.
\emph{REVSTAT-Statistical Journal}, \bold{18}, 5, 697–720.

To Duc, K., Chiogna, M. and Adimari, G. (2016)
Bias-corrected methods for estimating the receiver operating characteristic surface of continuous diagnostic tests.
\emph{Electronic Journal of Statistics}, \bold{10}, 3063-3113.

Guangming, P., Xiping, W. and Wang, Z. (2013)
Non-parameteric statistical inference for $P(X < Y < Z)$.
\emph{Sankhya A}, \bold{75}, 1, 118-138.
}
