#' Extracting list of uuid columns from biblioverlap results
#'
#' @description
#' Helper function to [plot_venn()] and [plot_upset()] that extracts overlapping data stored in the UUID column.
#'
#' @param db_list - list of matched dataframes (with UUID column added by biblioverlap)
#'
#' @return - a list containing only the UUID data
#' @noRd
#'
get_uuid_list <- function(db_list) {
  return( lapply(db_list, function(db) db$UUID) )
}


#' Plotting biblioverlap's matching summary
#'
#' @param matching_summary_df - summary of matching process generated by [biblioverlap()]
#' @param ... - additional arguments passed down to [ggplot2::geom_text()]
#'
#' @return a barplot summary of the matching results
#'
#' @importFrom rlang .data
#' @export
#'
#' @examples
#' #Running document-level matching procedure
#' biblioverlap_results <- biblioverlap(ufrj_bio_0122[1:2])
#'
#' #Checking biblioverlap results (summary table)
#' biblioverlap_results$summary
#'
#' #Plotting the matching summary
#' plot_matching_summary(biblioverlap_results$summary)
#'
plot_matching_summary <- function(matching_summary_df, ...) {
  plot <- matching_summary_df %>%
    ggplot2::ggplot(ggplot2::aes(x = .data$category, y = .data$n_docs, fill = .data$doc_subset)) +
    ggplot2::geom_bar(stat = 'identity', position = 'stack') +
    ggplot2::geom_text(ggplot2::aes(label = paste0(.data$n_docs," (",.data$perc_inside_category,"%)")), position = ggplot2::position_stack(vjust = 0.5), ... )

  return(plot)
}


#' Plotting Venn Diagram from biblioverlap results
#'
#' @param db_list - list of matched dataframes (with UUID column added by biblioverlap)
#' @param ... - Additional arguments that can be passed down to [ggVennDiagram::ggVennDiagram()]
#'
#' @return a Venn Diagram representation of document overlap between the input datasets
#' @export
#'
#' @examples
#' #Running document-level matching procedure
#' biblioverlap_results <- biblioverlap(ufrj_bio_0122[1:2])
#'
#' #Checking biblioverlap results (db_list)
#' lapply(biblioverlap_results$db_list, head, n=1)
#'
#' #Plotting the Venn diagram
#' plot_venn(biblioverlap_results$db_list)
#'
plot_venn <- function(db_list, ...) {
  uuid <- get_uuid_list(db_list)
  venn <- ggVennDiagram::ggVennDiagram(uuid, ...) +
    ggplot2::scale_fill_gradient(low = "#A7C7E7", high = "#08306B") +
    ggplot2::scale_x_continuous(expand = ggplot2::expansion(mult = .2))
  return ( venn )
}

#' Plotting UpSet plot from biblioverlap results
#'
#' @param db_list - list of matched dataframes (with UUID column added by biblioverlap)
#' @param ... - arguments to be passed down to [UpSetR::upset()]
#'
#' @return a UpSet plot representation of document overlap between the input datasets
#' @export
#'
#' @examples
#' #Running document-level matching procedure
#' biblioverlap_results <- biblioverlap(ufrj_bio_0122[1:2])
#'
#' #Checking biblioverlap results (db_list)
#' lapply(biblioverlap_results$db_list, head, n=1)
#'
#' #Plotting the UpSet plot
#' plot_upset(biblioverlap_results$db_list)
#'
plot_upset <- function(db_list, ...) {
  uuid <- get_uuid_list(db_list)
  upset <- UpSetR::upset(UpSetR::fromList(uuid), ...)
  return ( upset )
}


#' Shiny App for the biblioverlap package
#'
#' @param port - port of the application
#' @param max_upload_size - max upload size of documents (in MB) - Default 100
#' @param launch.browser - launch on browser - Default = TRUE
#'
#' @return opens a instance of the biblioverlap UI
#' @export
#'
#' @examples
#' #Running the ShinyApp
#' biblioverApp()
#'
biblioverApp <- function(port = NULL, max_upload_size = 1000, launch.browser = TRUE) {
  if (interactive()) {
    options(shiny.maxRequestSize = max_upload_size * 1024 * 1024)
    shiny::shinyAppDir(
      system.file("biblioverApp", package = "biblioverlap"),
      options = list(port = port, launch.browser = launch.browser)
    )
  } else {
    message("This function is intended for interactive sessions.")
    return(NULL)
  }
}
