% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/birdnet_interface.R
\name{predict_species_from_audio_file}
\alias{predict_species_from_audio_file}
\alias{predict_species_from_audio_file.birdnet_model}
\title{Predict species within an audio file using a BirdNET model}
\usage{
predict_species_from_audio_file(
  model,
  audio_file,
  min_confidence = 0.1,
  batch_size = 1L,
  chunk_overlap_s = 0,
  use_bandpass = TRUE,
  bandpass_fmin = 0L,
  bandpass_fmax = 15000L,
  apply_sigmoid = TRUE,
  sigmoid_sensitivity = 1,
  filter_species = NULL,
  keep_empty = TRUE,
  use_arrow = FALSE
)

\method{predict_species_from_audio_file}{birdnet_model}(
  model,
  audio_file,
  min_confidence = 0.1,
  batch_size = 1L,
  chunk_overlap_s = 0,
  use_bandpass = TRUE,
  bandpass_fmin = 0L,
  bandpass_fmax = 15000L,
  apply_sigmoid = TRUE,
  sigmoid_sensitivity = 1,
  filter_species = NULL,
  keep_empty = TRUE,
  use_arrow = FALSE
)
}
\arguments{
\item{model}{A BirdNET model object. An instance of the BirdNET model (e.g., \code{\link[=birdnet_model_tflite]{birdnet_model_tflite()}}).}

\item{audio_file}{character. The path to the audio file.}

\item{min_confidence}{numeric. Minimum confidence threshold for predictions (default is 0.1).}

\item{batch_size}{integer. Number of audio samples to process in a batch (default is 1L).}

\item{chunk_overlap_s}{numeric. The overlap between audio chunks in seconds (default is 0). Must be in the interval [0.0, 3.0].}

\item{use_bandpass}{logical. Whether to apply a bandpass filter (default is TRUE).}

\item{bandpass_fmin, bandpass_fmax}{integer. Minimum and maximum frequencies for the bandpass filter (in Hz). Ignored if \code{use_bandpass} is FALSE (default is 0L to 15000L).}

\item{apply_sigmoid}{logical. Whether to apply a sigmoid function to the model output (default is TRUE).}

\item{sigmoid_sensitivity}{numeric. Sensitivity parameter for the sigmoid function (default is 1). Must be in the interval [0.5, 1.5]. Ignored if \code{apply_sigmoid} is FALSE.}

\item{filter_species}{NULL, a character vector of length greater than 0, or a list where each element is a single non-empty character string. Used to filter the predictions. If NULL (default), no filtering is applied.}

\item{keep_empty}{logical. Whether to include empty intervals in the output (default is TRUE).}

\item{use_arrow}{logical. Whether to use Arrow for processing predictions (default is FALSE).}
}
\value{
A data frame with the following columns:
\describe{
\item{start}{Start time of the prediction interval.}
\item{end}{End time of the prediction interval.}
\item{scientific_name}{Scientific name of the predicted species.}
\item{common_name}{Common name of the predicted species.}
\item{confidence}{BirdNET’s confidence score for the prediction.}
}
}
\description{
Use a BirdNET model to predict species within an audio file. The model can be a TFLite model, a custom model, or a Protobuf model.
}
\details{
\subsection{Sigmoid Activation}{

When \code{apply_sigmoid = TRUE}, the raw logit scores from the linear classifier are passed
through a sigmoid function, scaling them into the range [0, 1]. This unitless confidence
score reflects BirdNET’s certainty in its prediction (it is not a direct probability of species presence).
Adjusting the \code{sigmoid_sensitivity} parameter modifies the score distribution:
\itemize{
\item Values \strong{< 1} tend to produce more extreme scores (closer to 0 or 1).
\item Values \strong{> 1} result in scores that are more moderate (centered around intermediate values).
For additional details on BirdNET confidence scores and guidelines for converting them to probabilities, see Wood & Kahl (2024).
}
}

\subsection{Apache Arrow optimization}{

By default, predictions from Python are converted to R using basic data structures. For large datasets using Apache Arrow (\code{use_arrow=TRUE}) can significantly improve performance by reducing memory usage during data conversion
and minimizing data copying between R and Python.

When to use Apache Arrow:
\itemize{
\item Large audio files (>20 minutes)
\item Low confidence thresholds (\code{min_confidence < 0.1})
\item Memory-constrained environments
\item Whenever you encounter an unusual long pause after inference. This is a sign that the data conversion is taking a long time.
}

Note that using Apache Arrow requires additional dependencies (\code{arrow} R package and \code{pyarrow} Python package).
You can install them manually using \code{\link[=install_arrow]{install_arrow()}}.
}
}
\examples{
\dontrun{
model <- birdnet_model_tflite(version = "v2.4", language = "en_us")
audio_file <- system.file("extdata", "soundscape.mp3", package = "birdnetR")
predictions <- predict_species_from_audio_file(model, audio_file, min_confidence = 0.1)
}

}
\references{
Wood, C. M., & Kahl, S. (2024). Guidelines for appropriate use of BirdNET scores and other detector outputs. Journal of Ornithology. https://doi.org/10.1007/s10336-024-02144-5
}
\seealso{
\code{\link[=read_labels]{read_labels()}} for more details on species filtering.

\code{\link[=birdnet_model_tflite]{birdnet_model_tflite()}}, \code{\link[=birdnet_model_protobuf]{birdnet_model_protobuf()}}, \code{\link[=birdnet_model_custom]{birdnet_model_custom()}}
}
