#' Write data to a c3d object
#'
#' Set new data to an existing c3d object.
#'
#' This is a basic helper function to allow the modification of data within the
#' `c3dr` package for later export. The function call updates the data (point
#' and/or analog) and the appropriate parameters and header sections. Note that
#' not all parameters can be updated based on insufficient information. For
#' example, when using `c3d_setdata()` for updating the point data, the point
#' label parameter gets updated (based on the column headers), but the point
#' label descriptions will be unmodified. This can create minor inconsistencies
#' in the resulting c3d object, which in the worst case can lead to corrupt data
#' after export with [c3d_write()]. If you plan heavy modifications of the data
#' before export make sure to manually check and update all relevant parameters
#' as well as the residual data after calling `c3d_setdata()`.
#'
#' @param x A `c3d` object to be modified.
#' @param newdata The new point data that should be written to the `c3d` object.
#'   Usually a data frame of the class `c3d_data` as it is generated by
#'   [c3d_data()]. Defaults to `NULL`, which means that the point data will
#'   remain unchanged. The new point data can be in any format (wide, long,
#'   longest), but take care that the conventions of the format are met (see
#'   [c3d_data()] for details).
#' @param newanalog The new analog data that should be written to the `c3d`
#'   object. Usually a data frame of the class `c3d_analog` as it is generated
#'   by [c3d_analog()]. Defaults to `NULL`, which means that the analog data
#'   will remain unchanged.
#'
#' @return The modified c3d object.
#'
#' @examples
#' # Import example data
#' d <- c3d_read(c3d_example())
#'
#' # remove last frame from point data and analog data (10 subframes for analog)
#' d_cut <- c3d_data(d)[-340, ]
#' a_cut <- c3d_analog(d)[-(3391:3400), ]
#'
#' # write the new c3d object
#' d_new <- c3d_setdata(d, newdata = d_cut, newanalog = a_cut)
#' d_new
#'
#' @export

c3d_setdata <- function(x, newdata = NULL, newanalog = NULL) {
  # input validation
  if (!inherits(x, "c3d")) stop("'x' needs to be a list of class 'c3d'.")
  # validation for newdata is inside create_newdata()
  # validation for newanalog is inside create_newanalog()

  # set new point data
  if (!is.null(newdata)) {
    nd <- create_newdata(newdata)
    # rewrite point data
    x$data <- nd[["data"]]
    # rewrite point labels
    x$parameters$POINT$LABELS <- nd[["labels"]]
    # rewrite number of points
    x$parameters$POINT$USED <- length(nd[["labels"]])
    x$header$npoints <- length(nd[["labels"]])
    # rewrite frame number
    x$parameters$POINT$FRAMES <- length(nd[["data"]])
    x$header$nframes <- length(nd[["data"]])
  }

  # set new analog data
  if (!is.null(newanalog)) {
    # get number of subframes
    n_subframes <- x$header$analogperframe
    na <- create_newanalog(newanalog, n_subframes)

    # rewrite analog data
    x$analog <- na[["data"]]
    # rewrite analog labels
    x$parameters$ANALOG$LABELS <- na[["labels"]]
    # rewrite number of analog channels
    x$header$nanalogs <- length(na[["labels"]])
    x$parameters$ANALOG$USED <- length(na[["labels"]])

    # warning if analog data is incomplete for subframes
    if (nrow(newanalog) %% n_subframes != 0) {
      warning("Analog data length is not a multiplier of point data length.")
    }
  }

  # warnings if data is not exactly compatible
  if (length(x$data) < length(x$analog)) {
    warning("Point data has less frames than analog data.")
  } else if (length(x$data) > length(x$analog)) {
    warning(
      "Point data has more frames than analog data. ",
      "This may lead to corrupt c3d files."
    )
  }
  x
}

#' Create new point data
#'
#' Internal function to convert point data from a data frame to a nested list
#' for use in a c3d object.
#'
#' @noRd
create_newdata <- function(newdata) {
  if (!inherits(newdata, "c3d_data")) {
    stop("'newdata' needs to be a data.frame of class 'c3d_data'.")
  } else if (nrow(newdata) == 0) {
    stop("'newdata' is an empty data.frame.")
  }
  # convert to wide data format if necessary
  if (inherits(newdata, "c3d_data_wide")) {
    d <- newdata
  } else {
    d <- c3d_convert(newdata, "wide")
  }

  # Get labels
  labels <- unique(factor(sub("_[xyz]$", "", colnames(d))))
  n_points <- length(labels)
  n_frames <- nrow(d)

  # Convert to matrix
  m <- as.matrix(d)

  # Create the empty list structure
  out <- replicate(
    n_frames,
    replicate(n_points, numeric(3), simplify = FALSE),
    simplify = FALSE
  )

  # Convert data frame to matrix for easier handling
  m <- as.matrix(d)
  dimnames(m) <- NULL

  # For each frame (row)
  for (i in 1:n_frames) {
    # For each point
    for (j in 1:n_points) {
      # Get the x,y,z indices for this point
      idx <- (j - 1) * 3 + 1:3
      # Assign the values
      out[[i]][[j]] <- m[i, idx]
    }
  }

  list(
    data = out,
    labels = as.character(labels)
  )
}

#' Create new analog data
#'
#' Internal function to convert analog data from a data frame to a nested list
#' for use in a c3d object.
#'
#' @noRd
create_newanalog <- function(newanalog, nperframe) {
  if (!inherits(newanalog, "c3d_analog")) {
    stop("'newanalog' needs to be a data.frame of class 'c3d_analog'.")
  } else if (nrow(newanalog) == 0) {
    stop("'newanalog' is an empty data.frame.")
  }
  if (!is.numeric(nperframe) || !(nperframe > 0)) {
    stop("'nperframe' needs to be a positive number")
  }
  # get frame id
  frame <- ceiling(seq_len(nrow(newanalog)) / nperframe)
  # Split data frame based on frames
  df_list <- split(newanalog, frame)
  # convert to list of matrices
  mlist <- lapply(df_list, function(sub_df) {
    mat <- as.matrix(sub_df)
    dimnames(mat) <- NULL
    mat
  })
  names(mlist) <- NULL
  out <- list(
    data = mlist,
    labels = colnames(newanalog)
  )
  out
}
