\name{convert_lab_units}
\alias{convert_lab_units}
\title{
Convert Common Medical Laboratory Measures Between US and SI Units
}
\description{
Converts measurements for common medical laboratory analytes (e.g., Glucose, Creatinine, Hemoglobin) between the US Conventional unit system (US) and the International System of Units (SI). The function includes a comprehensive lookup table of conversion factors and robust error checking.
}
\usage{
convert_lab_units(value, analyte, from_unit, to_unit)
}
\arguments{
\item{value}{
A single numeric value of the measurement to be converted. Must be finite and non-missing.
}
\item{analyte}{
A single character string specifying the substance being measured (e.g., "Glucose", "Creatinine"). Case-insensitive.
}
\item{from_unit}{
A single character string specifying the starting unit system: \code{"US"} (US Conventional) or \code{"SI"} (International System). Case-insensitive.
}
\item{to_unit}{
A single character string specifying the target unit system: \code{"US"} or \code{"SI"}. Case-insensitive.
}
}
\value{
If successful, returns a character vector of length two: \code{c(converted_value, target_unit_string)}. If an input error occurs, returns a character string containing a detailed error message.
}
\details{
The function uses a fixed set of multiplication factors (\eqn{F}) where \eqn{US \times F = SI}.

Supported Analytes and Unit Conversions:
\itemize{
\item \strong{Glucose}: mg/dL (\code{US})  mmol/L (\code{SI}). Factor: 0.0555
\item \strong{Creatinine}: mg/dL (\code{US})  umol/L (\code{SI}). Factor: 88.4
\item \strong{BUN} (Urea Nitrogen): mg/dL (\code{US})  mmol/L (\code{SI}). Factor: 0.357
\item \strong{Cholesterol}: mg/dL (\code{US})  mmol/L (\code{SI}). Factor: 0.0259
\item \strong{Triglycerides}: mg/dL (\code{US})  mmol/L (\code{SI}). Factor: 0.0113
\item \strong{Calcium}: mg/dL (\code{US})  mmol/L (\code{SI}). Factor: 0.25
\item \strong{Magnesium}: mg/dL (\code{US})  mmol/L (\code{SI}). Factor: 0.4114
\item \strong{Bilirubin}: mg/dL (\code{US})  umol/L (\code{SI}). Factor: 17.1
\item \strong{Sodium}: mEq/L (\code{US})  mmol/L (\code{SI}). Factor: 1.0
\item \strong{Potassium}: mEq/L (\code{US})  mmol/L (\code{SI}). Factor: 1.0
\item \strong{Hemoglobin}: g/dL (\code{US})  g/L (\code{SI}). Factor: 10.0
\item \strong{Albumin}: g/dL (\code{US})  g/L (\code{SI}). Factor: 10.0
\item \strong{Total Protein}: g/dL (\code{US})  g/L (\code{SI}). Factor: 10.0
\item \strong{Iron}: ug/dL (\code{US})  umol/L (\code{SI}). Factor: 0.1791
\item \strong{Ferritin}: ng/mL (\code{US})  pmol/L (\code{SI}). Factor: 2.247
}
}
\examples{

#1. Convert Glucose from US (mg/dL) to SI (mmol/L)

us_glucose <- 150
convert_lab_units(us_glucose, "Glucose", "US", "SI")

#2. Convert Creatinine from SI (umol/L) to US (mg/dL)

si_creatinine <- 110
convert_lab_units(si_creatinine, "Creatinine", "SI", "US")

#3. Handling analytes with a factor of 1 (Sodium)

convert_lab_units(140, "Sodium", "US", "SI")

#4. Error Case: Unsupported analyte

convert_lab_units(5, "TSH", "US", "SI")
}
\keyword{conversion}
\keyword{units}
\keyword{laboratory}
\keyword{clinical}
