% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ceplot.R
\name{ceplot}
\alias{ceplot}
\title{Interactive conditional expectation plot}
\usage{
ceplot(data, model, response = NULL, sectionvars = NULL,
  conditionvars = NULL, threshold = NULL, lambda = NULL,
  distance = c("euclidean", "maxnorm"), type = c("default", "separate",
  "shiny"), view3d = FALSE, Corder = "default", selectortype = "minimal",
  conf = FALSE, probs = FALSE, col = "black", pch = NULL,
  residuals = FALSE, xsplotpar = NULL, modelpar = NULL,
  xcplotpar = NULL)
}
\arguments{
\item{data}{A dataframe containing the data to plot}

\item{model}{A model object, or list of model objects}

\item{response}{Character name of response in \code{data}}

\item{sectionvars}{Character name of variable(s) from \code{data} on which to
take a section, can be of length 1 or 2.}

\item{conditionvars}{Character names of conditioning variables from
\code{data}. These are the predictors which we can set to single values in
order to produce a section. Can be a list of vectors of length 1 or 2. Can
be a character vector, which is then paired up using
\code{\link{arrangeC}}. If \code{NULL}, an attempt will be made to extract
all variable names which are not \code{response} or \code{sectionvars} from
\code{model}, and these will be arranged using \code{\link{arrangeC}}.}

\item{threshold}{This is a threshold distance. Points further than
\code{threshold} away from the current section will not be visible. Passed
to \code{\link{similarityweight}}.}

\item{lambda}{A constant to multiply by number of factor mismatches in
constructing a general dissimilarity measure. If left \code{NULL}, behaves
as though \code{lambda} is set greater than \code{threshold}, and so only
observations whose factor levels match the current section are visible.
Passed to \code{\link{similarityweight}}.}

\item{distance}{A character vector describing the type of distance measure to
use, either \code{"euclidean"} (default) or \code{"maxnorm"}.}

\item{type}{This specifies the type of interactive plot. \code{"default"}
places everything on one device. \code{"separate"} places condition
selectors on one device and the section on another. (These two options
require XQuartz on OS X). \code{"shiny"} produces a Shiny application.}

\item{view3d}{Logical; if \code{TRUE} plots a three-dimensional
regression surface if possible.}

\item{Corder}{Character name for method of ordering conditioning variables.
See \code{\link{arrangeC}}.}

\item{selectortype}{Type of condition selector plots to use. Must be
\code{"minimal"} if \code{type} is \code{"default"}. If \code{type} is
\code{"separate"}, can be \code{"pcp"} (see \code{\link{plotxc.pcp}}) or
\code{"full"} (see \code{\link{plotxc.full}}).}

\item{conf}{Logical; if \code{TRUE} plots confidence bounds (or equivalent)
for models which provide this.}

\item{probs}{Logical; if \code{TRUE}, shows predicted class probabilities
instead of just predicted classes. Only available if \code{S} specifies two
numeric predictors and the model's predict method provides this.}

\item{col}{Colour for observed data.}

\item{pch}{Plot symbols for observed data.}

\item{residuals}{Logical; if \code{TRUE}, plots a residual versus predictor
plot instead of the usual scale of raw response.}

\item{xsplotpar}{Plotting parameters for section visualisation as a list,
passed to \code{\link{plotxs}}. Can specify \code{xlim}, \code{ylim}.}

\item{modelpar}{Plotting parameters for models as a list, passed to
\code{\link{plotxs}}. Not used.}

\item{xcplotpar}{Plotting parameters for condition selector plots as a list,
passed to \code{\link{plotxc}}. Can specify \code{col} for highlighting
current section, \code{cex}, and \code{trim} (see \code{\link{plotxc}}).}
}
\description{
Creates an interactive conditional expectation plot, which
  consists of two main parts. One part is a single plot depicting a section
  through a fitted model surface, or conditional expectation. The other part
  shows small data summaries which give the current condition, which can be
  altered by clicking with the mouse.
}
\examples{
\dontrun{
## Example 1: Multivariate regression, xs one continuous predictor

mtcars$cyl <- as.factor(mtcars$cyl)

library(mgcv)
model1 <- list(
    quadratic = lm(mpg ~ cyl + hp + wt + I(wt^2), data = mtcars),
    additive = mgcv::gam(mpg ~ cyl + hp + s(wt), data = mtcars))

conditionvars1 <- list(c("cyl", "hp"))

ceplot(data = mtcars, model = model1, response = "mpg", sectionvars = "wt",
  conditionvars = conditionvars1, threshold = 0.3, conf = T)

## Example 2: Binary classification, xs one categorical predictor

mtcars$cyl <- as.factor(mtcars$cyl)
mtcars$am <- as.factor(mtcars$am)

library(e1071)
model2 <- list(
  svm = svm(am ~ mpg + wt + cyl, data = mtcars, family = "binomial"),
  glm = glm(am ~ mpg + wt + cyl, data = mtcars, family = "binomial"))

ceplot(data = mtcars, model = model2, sectionvars = "wt", threshold = 1,
  type = "shiny")

## Example 3: Multivariate regression, xs both continuous

mtcars$cyl <- as.factor(mtcars$cyl)
mtcars$gear <- as.factor(mtcars$gear)

library(e1071)
model3 <- list(svm(mpg ~ wt + qsec + cyl + hp + gear,
  data = mtcars, family = "binomial"))

conditionvars3 <- list(c("cyl","gear"), "hp")

ceplot(data = mtcars, model = model3, sectionvars = c("wt", "qsec"),
  threshold = 1, conditionvars = conditionvars3)

ceplot(data = mtcars, model = model3, sectionvars = c("wt", "qsec"),
    threshold = 1, type = "separate", view3d = T)

## Example 4: Multi-class classification, xs both categorical

mtcars$cyl <- as.factor(mtcars$cyl)
mtcars$vs <- as.factor(mtcars$vs)
mtcars$am <- as.factor(mtcars$am)
mtcars$gear <- as.factor(mtcars$gear)
mtcars$carb <- as.factor(mtcars$carb)

library(e1071)
model4 <- list(svm(carb ~ ., data = mtcars, family = "binomial"))

ceplot(data = mtcars, model = model4, sectionvars = c("cyl", "gear"),
  threshold = 3)

## Example 5: Multi-class classification, xs both continuous

data(wine)
wine$Class <- as.factor(wine$Class)
library(e1071)

model5 <- list(svm(Class ~ ., data = wine, probability = TRUE))

ceplot(data = wine, model = model5, sectionvars = c("Hue", "Flavanoids"),
  threshold = 3, probs = TRUE)

ceplot(data = wine, model = model5, sectionvars = c("Hue", "Flavanoids"),
  threshold = 3, type = "separate")

ceplot(data = wine, model = model5, sectionvars = c("Hue", "Flavanoids"),
  threshold = 3, type = "separate", selectortype = "pcp")

## Example 6: Multi-class classification, xs with one categorical predictor,
##            and one continuous predictor.

mtcars$cyl <- as.factor(mtcars$cyl)
mtcars$carb <- as.factor(mtcars$carb)

library(e1071)
model6 <- list(svm(cyl ~ carb + wt + hp, data = mtcars, family = "binomial"))

ceplot(data = mtcars, model = model6, threshold = 1, sectionvars = c("carb",
  "wt"), conditionvars = "hp")
}

}
\references{
O'Connell M, Hurley CB and Domijan K (2017). ``Conditional
  Visualization for Statistical Models: An Introduction to the
  \strong{condvis} Package in R.''\emph{Journal of Statistical Software},
  \strong{81}(5), pp. 1-20. <URL:http://dx.doi.org/10.18637/jss.v081.i05>.
}
\seealso{
\code{\link{condtour}}, \code{\link{similarityweight}}
}

