% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model-class.R
\name{simulate,ddm-method}
\alias{simulate,ddm-method}
\title{Simulate Data from a Decision Diffusion Model}
\usage{
\S4method{simulate}{ddm}(
  object,
  nsim = 4L,
  seed = NULL,
  n_subject = 3L,
  parameter_vector = NULL,
  time_parameters = c(t_min = -1, t_max = 1, dt = 0.001),
  debug = FALSE
)
}
\arguments{
\item{object}{An object of class \code{"ddm"} that defines the model
structure and parameters.}

\item{nsim}{Integer. Number of trials to simulate per subject. Defaults
to \code{4}. The function adjusts \code{nsim} if not divisible by
number of conditions.}

\item{seed}{Optional integer. Sets the random seed for reproducibility.
Defaults to \code{NULL}.}

\item{n_subject}{Integer. Number of subjects to simulate. Defaults to
\code{3}.}

\item{parameter_vector}{Optional. A named vector or list of parameters
(e.g., \code{a}, \code{z}, \code{v}, \code{t0}) used as the true value
to simulate. The function by default generates one or many true
parameter vector based on the population distribution stored in the
\code{ddm} object.}

\item{time_parameters}{Numeric vector of for setting the time range
and time step for simulation (using inverse transform method).}

\item{debug}{Logical. If \code{TRUE}, print debugging output during
simulation. Defaults to \code{FALSE}.}
}
\value{
A data frame with simulated trial data, including columns
like \code{subject}, \code{trial}, \code{choice}, and \code{rt}.
}
\description{
Simulate response times and choices from a Decision Diffusion model (DDM)
associated with a specific experimental design. The specification is
typically specified, using \code{ggdmcModel::BuildModel}).
}
\details{
This method simulates data from a design-based Decision Diffusion
model (DDM). You can simulate multiple subjects, override default
parameters, and use conduct inverse sampling with a fine time step.
You may turn on debugging output when something goes wrong.
}
\examples{
if (requireNamespace("ggdmcModel", quietly = TRUE)) {
  BuildModel <- getFromNamespace("BuildModel", "ggdmcModel")
  model <- ggdmcModel::BuildModel(
    p_map = list(
      a = "1", v = "1", z = "1", d = "1", sz = "1", sv = "1",
      t0 = "1", st0 = "1", s = "1", precision = "1"
    ),
    match_map = list(M = list(s1 = "r1", s2 = "r2")),
    factors = list(S = c("s1", "s2")),
    constants = c(d = 0, s = 1, st0 = 0, sv = 0, precision = 3),
    accumulators = c("r1", "r2"),
    type = "fastdm"
  )

# Simulate one subject ------------------
sub_model <- setDDM(model)
p_vector <- c(a = 1, sz = 0.25, t0 = 0.15, v = 2.5, z = .38)
dat <- simulate(sub_model,
  nsim = 256, parameter_vector = p_vector,
  n_subject = 1
)
# Simulate multiple subjects ------------------
# 1. First build a population distribution to sample many p_vector's

pop_mean <- c(a = 1, sz = 0.25, t0 = 0.15, v = 2.5, z = 0.38)
pop_scale <- c(a = 0.05, sz = 0.01, t0 = 0.02, v = .5, z = 0.01)
pop_dist <- ggdmcPrior::BuildPrior(
  p0    = pop_mean,
  p1    = pop_scale,
  lower = c(0, 0, 0, -10, 0),
  upper = rep(NA, model@npar),
  dists = rep("tnorm", model@npar),
  log_p = rep(FALSE, model@npar)
)

# 2. Enter the population distribution to a DDM model
pop_model <- setDDM(model, population_distribution = pop_dist)

# 3. simulate method will use the distribution to sample new p_vector's
# Note: You must enter sensible pop_mean, pop_scale and distribution.
hdat <- simulate(pop_model, nsim = 128, n_subject = 32)
}

}
