% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Dense2dFPCA.R
\name{Dense2dFPCA}
\alias{Dense2dFPCA}
\title{Two-Dimensional Functional Principal Component Analysis for dense repeated functional data.}
\usage{
Dense2dFPCA(
  X.age.year,
  n,
  num.years,
  num.ages,
  fpca.op = list(),
  pc.num = NULL
)
}
\arguments{
\item{X.age.year}{An n by (num.years x num.ages) input data matrix, such that the ith row of the matrix gives the observed values for the ith individual. The values in each row are sorted first by years (dimension 1) and then by ages (dimension 2) within each year.}

\item{n}{The sample size.}

\item{num.years}{Dimension 1}

\item{num.ages}{Dimension 2}

\item{fpca.op}{A list of options control parameters specified by \code{list(name=value)} for the two-dimesnional FPCA; check fdapace::FPCA for available control options and default settings.}

\item{pc.num}{A scalar denoting the maximum number of components to consider for the two-dimensional FPCA; default: chosen by FVE if NULL.}
}
\value{
A list containing the following fields:
\item{mu}{An num.ages by num.years matrix containing the bivariate mean function estimate.}
\item{pc.num}{A scalar denoting the selected number of components for the two-dimensional FPCA.}
\item{res.2D.FPCA}{A list containing the FPCA output for the fitted two-dimensional FPCA.}
\item{scores}{An n by pc.num matrix of the estimated scores, such that the ith row of the matrix comprises estimated scores for the ith individual.}
\item{eig}{An (num.years x num.ages) by pc.num matrix of the estimated product eigen functions. The estimated eigenfunctions in the otput eig are in the form of a vector rather than a matrix. For example, the first column in eig gives the first estimated eigenfunction such that \eqn{gamma(s,t) -> eig[ ( (s-1)*num.ages + t ), 1]} where LHS is the bivariate function in the usual form and RHS gives the corresponding element in the output vector. The rows are sorted first by years (dimension 1) and then by ages (dimension 2) within each year.}
\item{FVE}{A vector of length pc.num, indicating the fraction of total variance explained by each product function, with corresponding 'FVEthreshold'.}
}
\description{
Note: The code works for dense functional data observed on a regular grid, missing values are allowed.
}
\details{
The code works for dense functional data (with missing values), with density in both the direction of (age) dimension 2 and (year) dimension 1.
}
\examples{
\donttest{
n <- 100 ### sample size
N <- 100
num.ages <- 20 ### dimension 2
num.years <- 15 ### dimension 1
dense_grid <- seq(0,1,length=N)
Lt <- list()
Ly <- list()
for (i in 1:n) {
  Lt[[i]] <- dense_grid ### dense time grid
  y_temp <- matrix(0,num.ages,num.years)
  for (s in 1:num.ages) {
    for (t in 1:num.years) {
      y_temp[s,t] <- y_temp[s,t]+cos(Lt[[i]][t])+rnorm(1,0,0.5)
    }
  }
  Ly[[i]] <- y_temp ### dense functional data
}
X.age.year <- matrix(0,n,num.years*num.ages)
for (i in 1:n) {
  X.age.year[i,] <- as.vector(Ly[[i]]) ### data matrix
}
res <- Dense2dFPCA(X.age.year, n , 15, 20, fpca.op=NULL,pc.num=2)
# Basic output
res$mu
res$pc.num
res$res.2D.FPCA
res$eig
res$FVE
res$pc.num
cumsum(res$FVE)
}

}
\references{
\itemize{
\item \cite{Chen, K., Delicado, P., & Müller, H. G. (2017). Modelling function-valued stochastic processes, with applications to fertility dynamics. Journal of the Royal Statistical Society Series B: Statistical Methodology, 79(1), 177-196.}
\item \cite{Chen, K., & Müller, H. G. (2012). Modeling repeated functional observations. Journal of the American Statistical Association, 107(500), 1599-1609.}
\item \cite{Hall, P.,  Müller, H.G. and Wang, J.L. (2006). Properties of principal component methods for functional and longitudinal data analysis. Annals of Statistics, 34(3), 1493-1517.}
\item \cite{Yao, F., Müller, H. G., & Wang, J. L. (2005). Functional data analysis for sparse longitudinal data. Journal of the American statistical association, 100(470), 577-590.}
}
}
