% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mating.R
\name{gena.mating}
\alias{gena.mating}
\title{Mating}
\usage{
gena.mating(
  population,
  fitness,
  parents.n,
  method = "rank",
  par = NULL,
  self = FALSE,
  iter = NULL
)
}
\arguments{
\item{population}{numeric matrix which rows are chromosomes i.e. vectors of 
parameters values.}

\item{fitness}{numeric vector which \code{i}-th element is the value of 
\code{fn} at point \code{population[i, ]}.}

\item{parents.n}{even positive integer representing the number of parents.}

\item{method}{mating method to be used for selection of parents.}

\item{par}{additional parameters to be passed depending on the \code{method}.}

\item{self}{logical; if \code{TRUE} then chromosome may mate itself. 
Otherwise mating is allowed only between different chromosomes.}

\item{iter}{iteration number of the genetic algorithm.}
}
\value{
The function returns a list with the following elements:
\itemize{
\item \code{parents} - matrix which rows are parents. The number of
rows of this matrix equals to \code{parents.n} while the number of columns
is \code{ncol(population)}.
\item \code{fitness} - vector which i-th element is the fitness of the
i-th parent.
\item \code{ind} - vector which i-th element is the index of i-th
parent in population so \code{$parents[i, ]} equals to 
\code{population[ind[i], ]}.
}
}
\description{
Mating (selection) method (algorithm) to be used in the
genetic algorithm.
}
\details{
Denote \code{population} by \eqn{C} which \code{i}-th row 
\code{population[i, ]} is a chromosome \eqn{c_{i}} i.e. the vector of 
parameter values of the function being optimized \eqn{f(.)} that is
provided via \code{fn} argument of \code{\link[gena]{gena}}.
The elements of chromosome \eqn{c_{ij}} are genes representing parameters 
values. Argument \code{fitness} is a vector of function values at
corresponding chromosomes i.e. \code{fitness[i]} corresponds to
\eqn{f_{i}=f(c_{i})}. Total number of chromosomes in population
\eqn{n_{population}} equals to \code{nrow(population)}.

Mating algorithm determines selection of chromosomes that will become parents. 
During mating each iteration one of chromosomes become a parent until
there are \eqn{n_{parents}} (i.e. \code{parents.n}) parents selected.
Each chromosome may become a parent multiple times or not become a 
parent at all.

Denote by \eqn{c^{s}_{i}} the \eqn{i}-th of selected parents. Parents
\eqn{c^{s}_{i}} and \eqn{c^{s}_{i + 1}} form a pair that will further
produce a child (offspring), where \eqn{i} is odd.
If \code{self = FALSE} then for each pair of parents 
\eqn{(c_{i}^s, c_{i+1}^s)} it is insured that 
\eqn{c^{s}_{i} \ne c^{s}_{i + 1}} except the case when there are several 
identical chromosomes in population. However \code{self} is ignored 
if \code{method} is \code{"tournament"}, so in this case self-mating
is always possible.

Denote by \eqn{p_{i}} the probability of a chromosome to become a parent.
Remind that each chromosome may become a parent multiple times.
Probability \eqn{p_{i}\left(f_{i}\right)} is a function 
of fitness \eqn{f_{i}}. Usually this function is non-decreasing so 
more fitted chromosomes have higher probability of becoming a parent. 
There is also an intermediate value \eqn{w_{i}} called weight such that:
\deqn{p_{i}=\frac{w_{i}}{\sum\limits_{j=1}^{n_{population}}w_{j}}}
Therefore all weights \eqn{w_{i}} are proportional to corresponding 
probabilities \eqn{p_{i}} by the same factor (sum of weights).

Argument \code{method} determines particular mating algorithm to be applied.
Denote by \eqn{\tau} the vector of parameters used by the algorithm.
Note that \eqn{\tau} corresponds to \code{par}. The algorithm determines
a particular form of the \eqn{w_{i}\left(f_{i}\right)} function which 
in turn determines \eqn{p_{i}\left(f_{i}\right)}.

If \code{method = "constant"} then all weights and probabilities are equal:
\deqn{w_{i}=1 => p_{i}=\frac{1}{n_{population}}}

If \code{method = "rank"} then each chromosome receives a rank \eqn{r_{i}} 
based on the fitness \eqn{f_{i}} value. So if \code{j}-th chromosome is the
fittest one and \code{k}-th chromosome has the lowest fitness value then
\eqn{r_{j}=n_{population}} and  \eqn{r_{k}=1}. The relationship
between weight \eqn{w_{i}} and rank \eqn{r_{i}} is as follows:
\deqn{w_{i}=\left(\frac{r_{i}}{n_{population}}\right)^{\tau_{1}}}
The greater value of \eqn{\tau_{1}} the greater portion of probability will
be delivered to more fitted chromosomes. 
Default value is \eqn{\tau_{1} = 0.5} so \code{par = 0.5}.

If \code{method = "fitness"} then weights are calculated as follows:
\deqn{w_{i}=\left(f_{i} - 
                  \min\left(f_{1},...,f_{n_{population}}\right) + 
                  \tau_{1}\right)^{\tau_{2}}}
By default \eqn{\tau_{1}=10} and \eqn{\tau_{2}=0.5} i.e. 
\code{par = c(10, 0.5)}. There is a restriction \eqn{\tau_{1}\geq0}
insuring that expression in brackets is non-negative.

If \code{method = "tournament"} then  \eqn{\tau_{1}} (i.e. \code{par}) 
chromosomes will be randomly selected with equal probabilities and without 
replacement. Then the chromosome with the highest fitness 
(among these selected chromosomes) value will become a parent.
It is possible to provide representation of this algorithm via 
probabilities \eqn{p_{i}} but the formulas are numerically unstable.
By default \code{par = min(5, ceiling(parents.n * 0.1))}.

Validation and default values assignment for \code{par} is performed inside
\code{\link[gena]{gena}} function not in \code{\link[gena]{gena.mating}}.
It allows to perform validation a single time instead of repeating it
each iteration of genetic algorithm.

For more information on mating (selection) algorithms
please see Shukla et. al. (2015).
}
\examples{
# Consider the following fitness function
fn <- function(x)
{
  val <- x[1] * x[2] - x[1] ^ 2 - x[2] ^ 2
}

# Randomly initialize the population
set.seed(123)
pop.nulation <- 10
population <- gena.population(pop.n = pop.nulation,
                              lower = c(-5, -5), 
                              upper = c(5, 5))

# Calculate fitness of each chromosome
fitness <- rep(NA, pop.nulation)
for(i in 1:pop.nulation)
{
  fitness[i] <- fn(population[i, ])
}

# Perform mating to select parents
parents <- gena.mating(population = population,
                       fitness = fitness,
                       parents.n = pop.nulation,
                       method = "rank",
                       par = 0.8)
print(parents)

}
\references{
A. Shukla, H. Pandey, D. Mehrotra (2015). 
Comparative review of selection techniques in genetic algorithm.
\emph{2015 International Conference on Futuristic Trends on Computational 
Analysis and Knowledge Management (ABLAZE)}, 515-519,
<doi:10.1109/ABLAZE.2015.7154916>.
}
