#' Preparer of main references for automatic insertion in scientific articles
#'
#' The article_bib function prepares the metadata of the main scientific references to be automatically included in a scientific article. Initially, the user obtains the list of main references using the principal_lister function from the bibliorefer package.  Then, read the articles, choose the ones you prefer, and display the positions of these articles in the input parameter, position_artic, of the article_bib function. The article_bib function internally calls the gerard_lister and bibtex_lister functions. The gerard_lister function separates and organizes the metadata, of each of the articles in the list chosen by the user, and stores them in a dataframe. Then, the bibtex_lister function is applied to the set of metadata organized in the dataframe and converts it to the BibTex style. The output of the article_bib function is the file with the .bib extension, containing the list of references in BibTex format, prepared to be automatically included in the reference list of a scientific paper.
#'
#' @param input_date is a dataframe with the scientific production database obtained of colection WoS, Scopus and others
#' @param input_tam is the length of the dataframe with the main scientifics articles, obtained using package bibliorefer.
#' @param position_artic is a parameter that shows the positions of the articles in the main list, obtained using package bibliorefer, chosen to be included in the reference list of a scientific paper. If part of the list is used, the set of articles is presented through a sequence or a concatenated set. If the complete list is used, the complete sequence is created
#' @param total_list is the parameter that defines whether all articles from the main list, obtained using package bibliorefer, will be used or not. This parameter contains the logical values TRUE or FALSE. If the full list is used, the value is TRUE. Otherwise, if a part of the list is used, the value is FALSE
#' @param input_linkdateacess is the parameter that shows the list of links and access dates of the chosen scientific articles
#'
#' @return This function returns a file with the .bib extension, containing the list of references in bibtex format, prepared to be automatically included in the reference list of a scientific paper.
#' @importFrom utils read.csv2
#' @importFrom bibliorefer ajust_pagin principal_lister
#' @export
#'
#' @references
#' 1 - Aria, M. & Cuccurullo, C. (2017) bibliometrix: An R-tool for comprehensive science mapping analysis, Journal of Informetrics, 11(4), pp 959-975, Elsevier.
#' 2 - Bibliometric indicators to evaluate scientific activity. C García-Villar, J M García-Santos. Radiologia (Engl Ed). 2021 May-Jun;63(3):228-235.
#'
#' @examples
#'
#' # Example 1 - Concatenated position article
#'
#' # File of database
#' file_db <- system.file("extdata","example_databaseart.csv", package = "gerefer")
#'
#' file_base <- system.file("extdata","tabarticle_example1.csv", package = "gerefer")
#'
#' input_date <- example_databaseart(file_db)
#' input_tam <- 20
#' total_list <- FALSE
#' position_artic <- c(1,2,4,5,9,11,14,16,19,20)
#' input_linkdateacess <- basede_linkdate(file_base)
#'
#' #Calls the function article_bib
#' lister_bibtex <- article_bib(input_date, input_tam, position_artic,
#' total_list, input_linkdateacess)
#' lister_bibtex
#'
#'# Example 2 - Position article in initial sequence
#'
#' # File of database
#' file_db <- system.file("extdata","example_databaseart.csv", package = "gerefer")
#'
#' file_base <- system.file("extdata","tabarticle_example2.csv", package = "gerefer")
#'
#' # Parameters of the function
#' input_date <- example_databaseart(file_db)
#' input_tam <- 20
#' total_list <- FALSE
#' position_artic <- seq(1,10,1)
#' input_linkdateacess <- basede_linkdate(file_base)
#'
#' #Calls the function article_bib
#' lister_bibtex <- article_bib(input_date, input_tam, position_artic,
#' total_list, input_linkdateacess)
#' lister_bibtex
#'
#' # Example 3 - Position article in final sequence
#'
#' # File of database
#' file_db <- system.file("extdata","example_databaseart.csv", package = "gerefer")
#'
#' file_base <- system.file("extdata","tabarticle_example3.csv", package = "gerefer")
#'
#' # Parameters of the function
#' input_date <- example_databaseart(file_db)
#' input_tam <- 20
#' total_list <- FALSE
#' position_artic <- seq(11,20,1)
#' input_linkdateacess <- basede_linkdate(file_base)
#'
#' #Calls the function article_bib
#' lister_bibtex <- article_bib(input_date, input_tam, position_artic,
#' total_list, input_linkdateacess)
#' lister_bibtex
#'
#'
#'# Example 4 - Position article total sequence
#'
#' # File of database
#' file_db <- system.file("extdata","example_databaseart.csv", package = "gerefer")
#'
#' file_base <- system.file("extdata","tabela_acessototal.csv", package = "gerefer")
#'
#' # Parameters of the function
#' input_date <- example_databaseart(file_db)
#' input_tam <- 20
#' total_list <- TRUE
#' position_artic <- seq(1,20,1)
#' input_linkdateacess <- basede_linkdate(file_base)
#'
#' #Calls the function article_bib
#' lister_bibtex <- article_bib(input_date, input_tam, position_artic,
#' total_list, input_linkdateacess)
#' lister_bibtex
#'
#'
#'
article_bib <- function(input_date, input_tam, position_artic,
                        total_list, input_linkdateacess){

  # Data entry
  input_date <- input_date
  input_tam <- input_tam
  basede_linkdate <- input_linkdateacess


  # Generate the main list of articles
  lister_princ <- principal_lister(input_date,input_tam)
  lista_prin <- lister_princ[[1]]
  lista_prin <- as.data.frame(lista_prin)
  lista_principa <- lister_princ

  # Call the author team function
  equipe_autors <- function(lista_prin){

    # Separate the authors and calculate the length
    eq_autor <- lista_prin$Authors
    comp_autor <- length(eq_autor)

    # Call the function from the authors name
    minu_autor <- function(eq_autor){

      # Data entry
      nom_autsep <- eq_autor

      # Change the letter to lowercase
      autor_minu <- tolower(nom_autsep)

      # Separate words
      let_minu <- strsplit(autor_minu,"")

      # Select first letter of words
      prl_minu <- let_minu[[1]][1]

      # Capitalize the first letter
      prl_maiu <- toupper(prl_minu)

      # Replace the lowercase letter with the uppercase letter
      autor <- sub(prl_minu,prl_maiu,autor_minu)

      # Return the author's name
      return(autor)

    }

    # Call the function that shortens the name
    autor <- minu_autor(eq_autor)

    # Create variables
    nome_autores <- numeric(comp_autor)
    nom_autor <- numeric(comp_autor)
    tam_nom <- numeric(comp_autor)

    # Create the loop
    for(i in 1:comp_autor){

      #Separate the author's name
      nom_autor[i] <- strsplit(eq_autor[i],split = " ")
      nom_autsep <- strsplit(nom_autor[[i]],split = " ")

      # Number of words in the name and connection element
      tam_nom[i] <- length(nom_autsep)
      virgula <- ", "

      # Create the conditional
      if(tam_nom[i] <= 2){

        nom_autsep <- unlist(nom_autsep)
        ent_autor <- nom_autsep[1]
        autorsep <- minu_autor(ent_autor)
        nome_autor <- paste0(autorsep,virgula,nom_autsep[2])

      }else{

        nom_autsep <- strsplit(nom_autor[[i]],split = " ")
        nom_autsep <- unlist(nom_autsep)
        tam_nom[i] <- length(nom_autsep)
        indimax <- tam_nom[i]-1
        nomes_separ <- numeric(indimax)
        ent_autor <- numeric(tam_nom[i])

        for(indi in 1:indimax) {
          ent_autor <- nom_autsep[indi]
          autorsep <- minu_autor(ent_autor)
          nomes_separ[indi] <- autorsep
        }

        # Create variable and connection element
        nome_maior <- numeric(tam_nom[i])
        conexao1 <- " "
        conexao2 <- ", "

        # Create the first case
        nome_maior[1] <- paste0(nomes_separ[1],conexao1)
        nom_autor <- nome_maior[1]

        # Loop to create other cases
        for(inde in 2:(indimax-1)) {

          nome_maior[inde] <- paste0(nome_maior[inde-1],nomes_separ[inde],conexao1)
          nom_autor <- nome_maior[inde]

        }

        # Create the penultimate case
        nome_maior[indimax] <- paste0(nome_maior[inde],nomes_separ[indimax])
        nom_autor <- nome_maior[indimax]
        nom_autsep[tam_nom[i]]

        # Create the last case
        ind_fim <- tam_nom[i]
        nome_maior[ind_fim] <- paste0(nom_autor,conexao2,nom_autsep[tam_nom[i]])
        nom_autor <- nome_maior[ind_fim]
        nome_autor <- nom_autor
      }

      # Create the variable with the names of the authors
      nome_autores[i] <- nome_autor
    }

    # Create the set of author names
    autores <- nome_autores

    # Create positions of interest
    posicao_anos <- lista_prin$Year
    posicaoinfo <- which(posicao_anos != "")

    # Create author team sizes
    tamanhos <- lista_prin$`Number of authors`
    tamanho <- tamanhos[posicaoinfo]
    tamanho <- as.numeric(tamanho)

    # Calculate team list size
    comp <- length(posicaoinfo)

    # Create counter positions
    posicaocontc <- numeric(comp)
    posicaocont <- numeric(comp)

    # Create the first counter position
    posicaocontc[1] <- 0
    posicaocont[1] <- tamanho[1]

    #Create the loop for length and position counter
    for(i in 2:comp){

      posicaocontc[i] <- posicaocontc[i-1] + tamanho[i-1]
      posicaocont[i] <- posicaocont[i-1] + tamanho[i]

    }

    # Create variables
    equipesnomes <- numeric()

    # Create the loop
    for(j in 1:(comp-1)){

      # Create positions and groups of authors
      indice <- (posicaocontc[j]+1):posicaocontc[j+1]
      autor <- autores[indice]

      # Create the conditions
      if(tamanho[j] <= 1){

        equipesnom <- autor

      }else{

        conexao <- " and "
        equipenom <- numeric(tamanho[j])

        # Create team first name
        equipenom[1] <- paste0(autor[1],conexao)
        equipesnom <- equipenom[1]

        # Create team name sequence
        for(ind in 2:(tamanho[j]-1)) {

          equipenom[ind] <- paste0(equipenom[ind-1],
                                   autor[ind],conexao)
          equipesnom <- equipenom[ind]
        }

        # Create the last name of the sequence
        equipenom[tamanho[j]] <- paste0(equipenom[tamanho[j]-1],
                                        autor[tamanho[j]])
        equipesnom <- equipenom[tamanho[j]]
      }
      equipesnomes[j] <- equipesnom
    }

    # Create last position names
    indice <- (posicaocont[comp-1]+1):posicaocont[comp]
    autor <- autores[indice]

    # Create the conditions
    j <- comp

    if(tamanho[j] <= 1){

      equipesnom <- autor

    }else{

      conexao <- " and "
      equipenom <- numeric(tamanho[j])

      # Create first name
      equipenom[1] <- paste0(autor[1],conexao)
      equipesnom <- equipenom[1]

      # Create the last sequence
      for(ind in 2:tamanho[j]-1) {

        equipenom[ind] <- paste0(equipenom[ind-1],
                                 autor[ind],conexao)
        equipesnom <- equipenom[ind]
      }

      # Group the last name of the author team
      equipenom[tamanho[j]] <- paste0(equipenom[tamanho[j]-1],
                                      autor[tamanho[j]])
      equipesnom <- equipenom[tamanho[j]]
    }
    equipesnomes[j] <- equipesnom

    return(equipesnomes)

  }
  # Create the names of the author teams
  equipesnomes <- equipe_autors(lista_prin)

  ######
  # Call the function that generates the intermediate information
  gerad_lister <- function(input_date,input_tam,lista_prin,
                           lista_principa,equipesnomes,position_artic,
                           total_list, basede_linkdate){

    # Call the page adjustment function
    input_ent <- lista_principa
    paginacao <- ajust_pagin(input_ent)
    # Call the adjusted pages variable
    pagi_ajust <- paginacao$Paginas

    # Create the variables of interest
    linhas <- row.names(lista_prin)
    linhas <- as.numeric(linhas)
    ano_art <- lista_prin$Year
    posicao_art <- lista_prin$`Position of article`
    titu_art <- lista_prin$Article
    jornal_art <- lista_prin$Journal
    pagi_art <- lista_prin$Paginas
    volume_art <- lista_prin$Volumes
    doi_art <- lista_prin$Link

    # Create table with variables of interest
    tabela_vari <- cbind(linhas,ano_art,posicao_art,titu_art,
                         jornal_art,pagi_art,pagi_ajust,volume_art,doi_art)
    tabela_vari <- as.data.frame(tabela_vari)

    # Create positions for the reduced list and reduced table
    anos <- lista_prin$Year
    posicao_info <- which(anos != "")



    ##
    tabela_infor <- tabela_vari[posicao_info,]

    # Create the position table
    tabela_info <- cbind(tabela_infor$linhas,tabela_infor$ano_art,
                         tabela_infor$posicao_art,equipesnomes,
                         tabela_infor$titu_art,tabela_infor$jornal_art,
                         tabela_infor$pagi_art,tabela_infor$pagi_ajust,
                         tabela_infor$volume_art,tabela_infor$doi_art)
    tabela_info <- as.data.frame(tabela_info)
    colnames(tabela_info) <- c("linhas","ano_art","posicao_art",
                               "equipesnomes","titu_art","jornal_art",
                               "pagi_art","pagi_ajust","volume_art",
                               "doi_art")

    # Parameter input
    art_esc <- position_artic
    parame_lister <- total_list
    n <- length(art_esc)

    # Take the test to create a list of titles
    if(parame_lister == F|FALSE){

      # Create the list of titles
      sequenc_artic <- art_esc
      tabela_inforedu <- tabela_info[sequenc_artic,]
      titu_art <- tabela_inforedu$titu_art
      jornal_art <- tabela_inforedu$jornal_art

    }else{

      # Create the list of titles
      sequenc_artic <- art_esc
      tabela_inforedu <- tabela_info
      titu_art <- tabela_info$titu_art
      jornal_art <- tabela_info$jornal_art
    }

    ######
    # Create positions of interest
    posicao_anos <- lista_prin$Year
    posicaoinfo <- which(posicao_anos != "")
    posicao_links <- lista_prin$Link[posicaoinfo]
    link_esc <- posicao_links[art_esc]

    # Create the table
    tabela_esc <- cbind(art_esc,link_esc,basede_linkdate$acess_date)
    tabela_esc <- as.data.frame(tabela_esc)
    nomes_esc <- c("art_esc","link_esc","date_esc")
    colnames(tabela_esc) <- nomes_esc

    # Call the function
    acesso_dialink <- function(tabela_esc,basede_linkdate,n){

      link_entra <- numeric(n)
      posi_entra <- numeric(n)
      acedat_entra <- numeric(n)

      # Delete NA
      na_retir <- function(input_entr){

        # Data entry
        entrada_base <- input_entr
        entrada_base <- as.data.frame(entrada_base)
        colnames(entrada_base) <- "base"

        # Delete NA
        variavel <- entrada_base$base
        testena <- is.na(variavel)
        padrao <- "TRUE"
        linha_na <- grep(padrao,testena)
        variavel[linha_na] <- ""

        return(variavel)

      }

      # Creation of the loop
      for(i in 1:n){

        # Treatment of article links
        link_ent <- basede_linkdate$link_article[i]
        teste_linkna <- is.na(link_ent)

        if(link_ent == "" | teste_linkna == TRUE){

          link_entra[i] <- na_retir(link_ent)

        }else{

          link_entra[i] <- link_esc[i]

        }

        # Treatment of article positions
        acedat_ent <- basede_linkdate$acess_date[i]
        teste_acedatna <- is.na(acedat_ent)

        if(acedat_ent == "" | teste_acedatna == TRUE){

          acedat_entra[i] <- na_retir(acedat_ent)

        }else{

          acedat_entra[i] <- acedat_ent

        }

      }

      link_article <- numeric(n)
      data_acesso <- numeric(n)
      doi_arti <- numeric(n)

      # Criação do laço
      for(i in 1:n){

        link <- link_entra[i]
        dat_ace <- acedat_entra[i]
        teste_chalink <- nchar(link)
        teste_chalinkna <- is.na(nchar(link))
        teste_chadate <- nchar(dat_ace)

        if(teste_chalink == 0 && teste_chalinkna == FALSE){

          link_articl <- tabela_esc$link_esc[i]

        }else if(teste_chalinkna == TRUE){

          link_articl <-  basede_linkdate$link_article[i]

        }else{

          link_articl <-  basede_linkdate$link_article[i]

        }

        if(teste_chadate == 0){

          datas_acess <- "01/02/25"

        }else{

          #link_articl <- link
          datas_acess <- tabela_esc$date_esc[i]
        }

        link_article[i] <- link_articl
        data_acesso[i] <- datas_acess
        doi_arti[i] <- ""

      }

      plani_dataacess <- cbind(doi_arti,link_article,data_acesso)
      plani_dataacess <- as.data.frame(plani_dataacess)

      return(plani_dataacess)

    }

    datasacesso <- acesso_dialink(tabela_esc,basede_linkdate,n)

    ##

    tamanho_titul <- n
    tamanho <- n
    linhas <- 1:tamanho
    data_separ <- numeric(tamanho)

    for(i in 1:tamanho){

      data_sepa <- strsplit(datasacesso$data_acesso, split = "/")
      data_separ <- data_sepa[[i]]

    }

    linhas <- 1:tamanho

    # Chamar a função da data de acesso
    acesso_datas <- function(data_separ){

      # Criar o dia
      ind_dia <- 1
      dia_separ <- data_separ[ind_dia]

      # Criar o mês
      ind_mes <- 2
      mes_separ <- data_separ[ind_mes]

      # Criar o ano
      ind_ano <- 3
      ano_sepa <- data_separ[ind_ano]
      ano_sepa <- as.numeric(ano_sepa)
      inicio_ano <- 20
      ano_separ <- paste0(inicio_ano,ano_sepa)

      # Gerar a data de acesso
      hifen <- "\u002D"
      data_acesso <- paste0(ano_separ, hifen, mes_separ, hifen, dia_separ)

      return(data_acesso)

    }

    data_acesso <- acesso_datas(data_separ)
    dia_acesso  <- numeric(tamanho)
    links <- numeric(tamanho)

    for(i in 1:tamanho){

      vazio <- ""
      di_acess <- datasacesso$data_acesso[i]
      li_acess <- datasacesso$link_article[i]

      if(di_acess == vazio){

        dia_acesso[i] <- ""
        links[i] <- ""

      }else{

        data_sepa <- strsplit(datasacesso$data_acesso, split = "/")
        data_separ <- data_sepa[[i]]
        data_acesso <- acesso_datas(data_separ)
        dia_acesso[i] <- data_acesso
        links[i] <- li_acess

      }

    }

    tamanho_subti <- numeric(tamanho_titul)
    titul_artic <- numeric(tamanho_titul)
    subtitul_artic <- numeric(tamanho_titul)

    # Call the function from the authors name
    titulo_minu <- function(enti_artic){

      # Data entry
      tit_arti <- enti_artic

      # Change the letter to lowercase
      titu_minu <- tolower(tit_arti)

      # Separate words
      let_minu <- strsplit(titu_minu," ")

      # Select first letter of words
      prl_min <- let_minu[[1]][1]

      # Separar letra da primeira palavra
      prl_minsep <- strsplit(prl_min, "")

      # Selecionar a primeira letra minúscula
      prl_minu <- prl_minsep[[1]][1]

      #"‘"=2018
      padrao_aspa1 <- "\u2018"
      test_aspa1 <- grepl(padrao_aspa1, prl_minu)

      #"“"=201C
      padrao_aspa2 <- "\u201C"
      test_aspa2 <- grepl(padrao_aspa2, prl_minu)

      #"'"=0027
      padrao_aspavert1 <- "\u0027"
      test_aspavert1 <- grepl(padrao_aspavert1, prl_minu)

      #"`"=0060
      padrao_apost1 <- "\u0060"
      test_apost1 <- grepl(padrao_apost1, prl_minu)

      #"\""="\0022"
      padrao_apost2 <- "\u0022"
      test_apost2 <- grepl(padrao_apost2, prl_minu)

      #"\\("=0028
      padrao_paren <- "\\u0028"
      test_paren <- grepl(padrao_paren, prl_minu)

      #"#"=0023
      padrao_hash <- "\u0023"
      test_hash <- grepl(padrao_hash, prl_minu)

      if(test_aspa1 == TRUE){

        prl_minu <- prl_minsep[[1]][2]

      }else if(test_aspa2 == TRUE){

        prl_minu <- prl_minsep[[1]][2]

      }else if(test_apost1 == TRUE){

        prl_minu <- prl_minsep[[1]][2]

      }else if(test_apost2 == TRUE){

        prl_minu <- prl_minsep[[1]][2]

      }else if(test_aspavert1 == TRUE){

        prl_minu <- prl_minsep[[1]][2]

      }else if(test_paren == TRUE){

        prl_minu <- prl_minsep[[1]][2]

      }else if(test_hash == TRUE){

        prl_minu <- prl_minsep[[1]][2]

      }else{

        prl_minu <- prl_minsep[[1]][1]

      }

      # Capitalize the first letter
      prl_maiu <- toupper(prl_minu)

      # Replace the lowercase letter with the uppercase letter
      tit_artic <- sub(prl_minu,prl_maiu,titu_minu)

      # Return the author's name
      return(tit_artic)

    }

    entitu_artic <- titu_art

    for(i in 1:tamanho){

      enti_artic <- entitu_artic[i]

      # Return the author's name
      padrao_pont <- ": "
      padrao_pont1 <- ":"
      test_subtitu <- grepl(padrao_pont,enti_artic)

      padrao_true <- "TRUE"
      linha_sub <- grep(padrao_true,test_subtitu)
      tamanho_titul <- length(linha_sub)

      # Standardize the page separator
      if(test_subtitu == FALSE){

        titu_artic <- enti_artic
        titu_artic <- titulo_minu(titu_artic)
        subtitu_artic <- ""

      }else{

        titulo_basesepar <- strsplit(enti_artic,
                                     split = padrao_pont)
        comp_subti <- length(titulo_basesepar[[1]])

        if(comp_subti == 2){

          titu_artic <- paste0(titulo_basesepar[[1]][1],padrao_pont1)
          titu_artic <- titulo_minu(titu_artic)
          subtitu_artic <- titulo_basesepar[[1]][2]
          subtitu_artic <- tolower(subtitu_artic)

        }else{

          titu_artic <- paste0(titulo_basesepar[[1]][1],padrao_pont1)
          titu_artic <- titulo_minu(titu_artic)
          subtitu_artic <- titulo_basesepar[[1]][2]

          for(k in 3:comp_subti){

            subtitu_artic <- paste0(subtitu_artic, sep = ".",
                                    titulo_basesepar[[1]][k])
            subtitu_artic <- tolower(subtitu_artic)
          }

        }

      }

      titul_artic[i] <- titu_artic
      subtitul_artic[i] <- subtitu_artic

    }

    ###
    # Function that changes the title to lowercase
    # Calculate variable sizes

    journal_maius <- numeric(n)
    jornais <- numeric(n)
    # Create new variable for titles
    journal_art <- jornal_art

    # Create the loop
    for(i in 1:n){

      # Input with capital letter
      entrada_journal <- journal_art[i]

      # Entry indexed with capital letter
      journal_maius[i] <- entrada_journal

      # Change the format to lowercase
      journal_minu <- tolower(entrada_journal)

      # Separate words
      journal_separ <- strsplit(journal_minu," ")

      # Select first word
      primei_palajour <- journal_separ[[1]][1]

      # Separate letters from words
      pljour <- strsplit(primei_palajour,"")

      # Select first letter of words
      primeira_palajour <- pljour[[1]][1]

      # Capitalize the first letter
      primeira_maiujour <- toupper(primeira_palajour)

      # Replace the lowercase letter with the uppercase letter
      jour <- sub(primeira_palajour,primeira_maiujour,journal_minu)

      # Create indexed variable
      jornais[i] <- jour
    }

    #####
    # Create variables for the table
    doi_artigo <- datasacesso$doi_arti

    # Create the reduced table
    tabela_reduzi <- cbind(tabela_inforedu$linhas, tabela_inforedu$ano_art,
                           tabela_inforedu$posicao_art, tabela_inforedu$equipesnomes,
                           titul_artic, subtitul_artic, jornais, tabela_inforedu$pagi_art,
                           tabela_inforedu$pagi_ajust,tabela_inforedu$volume_art,
                           doi_artigo,links,dia_acesso)
    tabela_reduzi <- as.data.frame(tabela_reduzi)
    colnames(tabela_reduzi) <- c("linhas", "ano_art", "posicao_art", "equipesnomes",
                                 "titu_art", "subtitu_art", "jornal_art", "pagi_art",
                                 "pagi_ajust", "volume_art", "doi_art", "link_art",
                                 "dia_acesso")

    ###
    # Create the year author identity
    # Separate team authors
    prim_aut <- strsplit(tabela_reduzi$equipesnomes,
                         split = " and ")

    # Create storage vectors
    primei_aut <- numeric(n)
    primei_auto <- numeric(n)
    primei_autor <- numeric(n)
    autor_ano <- numeric(n)

    # Create the loop to generate the author year identity
    for(i in 1:n){

      primei_aut[i] <- prim_aut[[i]][1]
      primei_auto[i] <- strsplit(primei_aut[i],split = ",")
      primei_autor[i] <- primei_auto[[i]][1]
      autor_ano[i] <- paste0(primei_autor[i],
                             tabela_reduzi$ano_art[i])
    }

    # Modify the reduced table
    autor_ano <- cbind(autor_ano)
    tabelaf <- cbind(autor_ano,tabela_reduzi)
    tabelaf <- as.data.frame(tabelaf)

    tabel_bibtex <- cbind(tabelaf$autor_ano, tabelaf$ano_art, tabelaf$equipesnomes,
                          tabelaf$titu_art, tabelaf$subtitu_art, tabelaf$jornal_art,
                          tabelaf$pagi_art,tabelaf$pagi_ajust,tabelaf$volume_art,
                          tabelaf$doi_art,links,dia_acesso)

    # # Modify column names and format
    tabel_bibtex <- as.data.frame(tabel_bibtex)
    colnames(tabel_bibtex) <- c("fonte_ano","ano","autores","titulos","subtitulos",
                                "jornais","paginas","paginasaju","volumes","doi",
                                "links","dataacesso")

    # Returns information in bibtex format
    return(tabel_bibtex)

  }
  # Call the function
  tabel_bibtex <- gerad_lister(input_date,input_tam, lista_prin,lista_principa,
                               equipesnomes, position_artic, total_list,
                               basede_linkdate)
  ########################################################
  # Function that generates the file in bibtex format
  bibtex_lister <- function(tabel_bibtex,indic){

    # Data entry
    input_tabel <- tabel_bibtex
    tabela <- input_tabel
    linha <- indic

    # Reduce database size
    tabela_redu <- tabela[linha,]
    tabela_ent <- tabela_redu

    # Assign input information to variables
    simbolo_entrada <- "@article{"
    fonte_ano <- tabela_ent$fonte_ano
    titulos <- tabela_ent$titulos
    subtitulos <- tabela_ent$subtitulos
    autores <- tabela_ent$autores
    ano <- tabela_ent$ano
    revista <- tabela_ent$jornais
    pagina <- tabela_ent$paginasaju
    volume <- tabela_ent$volumes
    doi <- tabela_ent$doi
    link <- tabela_ent$links
    acesso <- tabela_ent$dataacesso

    # Create the elements of the bibtex format structure
    espaco <- "  "
    virgula <- ","
    tituloarticle <- "title="
    subtituloarticle <- "subtitle="
    autorarticle <- "author="
    anoarticle <- "year="
    revistaarticle <- "journal="
    paginaarticle <- "pages="
    volumearticle <- "volume="
    doiarticle <- "doi="
    linkarticle <- "url="
    acessoarticle <- "urldate="
    chave_entrada <- "{"
    chave_saida <- "}"

    # Create the bibtex structure for each variable
    chave <- paste0(simbolo_entrada,fonte_ano,virgula)
    titulo <- paste0(tituloarticle,chave_entrada,titulos,
                     chave_saida)
    subtitulo <- paste0(subtituloarticle,chave_entrada,subtitulos,
                        chave_saida)
    autor <- paste0(autorarticle,chave_entrada,autores,
                    chave_saida)
    ano <- paste0(anoarticle,chave_entrada,ano,
                  chave_saida)
    revista <- paste0(revistaarticle,chave_entrada,revista,
                      chave_saida)
    pagina <- paste0(paginaarticle,chave_entrada,pagina,
                     chave_saida)
    volume <- paste0(volumearticle,chave_entrada,volume,
                     chave_saida)
    doi <- paste0(doiarticle,chave_entrada,doi,
                  chave_saida)
    link <- paste0(linkarticle,chave_entrada,link,
                   chave_saida)
    acesso <- paste0(acessoarticle,chave_entrada,acesso,
                     chave_saida)

    # Create bibtex format for articles
    referen_bibtex <- paste0(chave,"\n",
                             espaco, titulo, virgula, "\n",
                             espaco, subtitulo, virgula, "\n",
                             espaco, autor, virgula, "\n",
                             espaco, ano, virgula, "\n",
                             espaco, revista, virgula, "\n",
                             espaco, pagina, virgula, "\n",
                             espaco, volume, virgula, "\n",
                             espaco, doi, virgula, "\n",
                             espaco, link, virgula, "\n",
                             espaco, acesso, "\n", chave_saida,
                             espaco, collapse = ",\n")

    # Returns the file in bibtex format
    return(referen_bibtex)
  }

  # Call the function
  tabel_bibtex <- gerad_lister(input_date,input_tam,lista_prin,lista_principa,
                               equipesnomes,position_artic,total_list,
                               basede_linkdate)

  # Calculate file list size
  nlista <- length(position_artic)

  # Create the variable to store values
  arquivo_bibtex <- c()

  # Create the loop to generate the list of bibtex files
  for(i in 1:nlista){

    # Create the parameter linked to the index
    indic <- i

    # Call the function that converts to bibtex format
    refer_bibtex <- bibtex_lister(tabel_bibtex,indic)
    arquivo_bibtex[i] <- refer_bibtex

  }

  # Return the value of the variable
  lista_final <- arquivo_bibtex

  return(lista_final)

}
