% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/find_MAP.R
\name{find_MAP}
\alias{find_MAP}
\title{Find the Maximum A Posteriori Estimation}
\usage{
find_MAP(
  g,
  max_iter = NA,
  optimizer = NA,
  show_progress_bar = TRUE,
  save_all_perms = FALSE,
  return_probabilities = FALSE
)
}
\arguments{
\item{g}{Object of a \code{gips} class.}

\item{max_iter}{The number of iterations for an algorithm to perform.
At least 2. For \code{optimizer = "BF"}, it is not used;
for \code{optimizer = "MH"}, it has to be finite;
for \code{optimizer = "HC"}, it can be infinite.}

\item{optimizer}{The optimizer for the search of the maximum posteriori:
\itemize{
\item \code{"BF"} (the default for unoptimized \code{g} with \verb{perm size <= 9}) - Brute Force;
\item \code{"MH"} (the default for unoptimized \code{g} with \verb{perm size > 10}) - Metropolis-Hastings;
\item \code{"HC"} - Hill Climbing;
\item \code{"continue"} (the default for optimized \code{g}) - The same as
the \code{g} was optimized by (see Examples).
}

See the \strong{Possible algorithms to use as optimizers}
section below for more details.}

\item{show_progress_bar}{A boolean.
Indicate whether or not to show the progress bar:
\itemize{
\item When \code{max_iter} is infinite, \code{show_progress_bar} has to be \code{FALSE};
\item When \code{return_probabilities = TRUE}, then
shows an additional progress bar for the time
when the probabilities are calculated.
}}

\item{save_all_perms}{A boolean. \code{TRUE} indicates saving
a list of all permutations visited during optimization.
This can be useful sometimes but needs a lot more RAM.}

\item{return_probabilities}{A boolean. \code{TRUE} can only be provided
only when \code{save_all_perms = TRUE}. For:
\itemize{
\item \code{optimizer = "MH"} - use Metropolis-Hastings results to
estimate posterior probabilities;
\item \code{optimizer = "BF"} - use brute force results to
calculate exact posterior probabilities.
}

These additional calculations are costly, so a second and third
progress bar is shown (when \code{show_progress_bar = TRUE}).

To examine probabilities after optimization,
call \code{\link[=get_probabilities_from_gips]{get_probabilities_from_gips()}}.}
}
\value{
Returns an optimized object of a \code{gips} class.
}
\description{
Use one of the optimization algorithms to find the permutation that
maximizes a posteriori probability based on observed data.
Not all optimization algorithms will always find the MAP, but they try
to find a significant value. More information can be found in
the "\strong{Possible algorithms to use as optimizers}" section below.
}
\details{
\code{find_MAP()} can produce a warning when:
\itemize{
\item the optimizer "hill_climbing" gets to the end of
its \code{max_iter} without converging.
\item the optimizer will find the permutation with smaller \code{n0} than
\code{number_of_observations} (for more information on what it means,
see \strong{\eqn{C_\sigma} and \code{n0}} section
in the \code{vignette("Theory", package = "gips")} or in its
\href{https://przechoj.github.io/gips/articles/Theory.html}{pkgdown page}).
}
}
\section{Possible algorithms to use as optimizers}{


For an in-depth explanation, see in the
\code{vignette("Optimizers", package = "gips")} or in its
\href{https://przechoj.github.io/gips/articles/Optimizers.html}{pkgdown page}.

For every algorithm, there are some aliases available.
\itemize{
\item \code{"brute_force"}, \code{"BF"}, \code{"full"} - use
the \strong{Brute Force} algorithm that checks the whole permutation
space of a given size. This algorithm will find
the actual Maximum A Posteriori Estimation, but it is
very computationally expensive for bigger spaces.
We recommend Brute Force only for \code{p <= 9}.
For the time the Brute Force takes on our machines, see in the
\code{vignette("Optimizers", package = "gips")} or in its
\href{https://przechoj.github.io/gips/articles/Optimizers.html}{pkgdown page}.
\item \code{"Metropolis_Hastings"}, \code{"MH"} - use
the \strong{Metropolis-Hastings} algorithm;
\href{https://en.wikipedia.org/wiki/Metropolis\%E2\%80\%93Hastings_algorithm}{see Wikipedia}.
The algorithm will draw a random transposition in every iteration
and consider changing the current state (permutation).
When the \code{max_iter} is reached, the algorithm will return the best
permutation calculated as the MAP Estimator. This implements
the \href{https://arxiv.org/abs/2004.03503}{\emph{Second approach} from references, section 4.1.2}.
This algorithm used in this context is a special case of the
\strong{Simulated Annealing} the user may be more familiar with;
\href{https://en.wikipedia.org/wiki/Simulated_annealing}{see Wikipedia}.
\item \code{"hill_climbing"}, \code{"HC"} - use
the \strong{hill climbing} algorithm;
\href{https://en.wikipedia.org/wiki/Hill_climbing}{see Wikipedia}.
The algorithm will check all transpositions in every iteration and
go to the one with the biggest a posteriori value.
The optimization ends when all \emph{neighbors} will have a smaller
a posteriori value. If the \code{max_iter} is reached before the end,
then the warning is shown, and it is recommended to continue
the optimization on the output of the \code{find_MAP()} with
\code{optimizer = "continue"}; see examples.
Remember that \code{p*(p-1)/2} transpositions will be checked
in every iteration. For bigger \code{p}, this may be costly.
}
}

\examples{
require("MASS") # for mvrnorm()

perm_size <- 5
mu <- runif(perm_size, -10, 10) # Assume we don't know the mean
sigma_matrix <- matrix(
  data = c(
    1.0, 0.8, 0.6, 0.6, 0.8,
    0.8, 1.0, 0.8, 0.6, 0.6,
    0.6, 0.8, 1.0, 0.8, 0.6,
    0.6, 0.6, 0.8, 1.0, 0.8,
    0.8, 0.6, 0.6, 0.8, 1.0
  ),
  nrow = perm_size, byrow = TRUE
) # sigma_matrix is a matrix invariant under permutation (1,2,3,4,5)
number_of_observations <- 13
Z <- MASS::mvrnorm(number_of_observations, mu = mu, Sigma = sigma_matrix)
S <- cov(Z) # Assume we have to estimate the mean

g <- gips(S, number_of_observations)

g_map <- find_MAP(g, max_iter = 5, show_progress_bar = FALSE, optimizer = "Metropolis_Hastings")
g_map

g_map2 <- find_MAP(g_map, max_iter = 5, show_progress_bar = FALSE, optimizer = "continue")

if (require("graphics")) {
  plot(g_map2, type = "both", logarithmic_x = TRUE)
}

g_map_BF <- find_MAP(g, show_progress_bar = FALSE, optimizer = "brute_force")
summary(g_map_BF)
}
\references{
Piotr Graczyk, Hideyuki Ishi, Bartosz Kołodziejek, Hélène Massam.
"Model selection in the space of Gaussian models invariant by symmetry."
The Annals of Statistics, 50(3) 1747-1774 June 2022.
\href{https://arxiv.org/abs/2004.03503}{arXiv link};
\doi{10.1214/22-AOS2174}
}
\seealso{
\itemize{
\item \code{\link[=gips]{gips()}} - The constructor of a \code{gips} class.
The \code{gips} object is used as the \code{g} parameter of \code{find_MAP()}.
\item \code{\link[=plot.gips]{plot.gips()}} - Practical plotting function for
visualizing the optimization process.
\item \code{\link[=summary.gips]{summary.gips()}} - Summarize the output of optimization.
\item \code{\link[=AIC.gips]{AIC.gips()}}, \code{\link[=BIC.gips]{BIC.gips()}} - Get the Information Criterion
of the found model.
\item \code{\link[=get_probabilities_from_gips]{get_probabilities_from_gips()}} - When
\code{find_MAP(return_probabilities = TRUE)} was called,
probabilities can be extracted with this function.
\item \code{\link[=log_posteriori_of_gips]{log_posteriori_of_gips()}} - The function that the optimizers
of \code{find_MAP()} tries to find the argmax of.
\item \code{\link[=forget_perms]{forget_perms()}} - When the \code{gips} object was optimized
with \code{find_MAP(save_all_perms = TRUE)}, it will be of
considerable size in RAM. \code{forget_perms()} can make such an object
lighter in memory by forgetting the permutations it visited.
\item \code{vignette("Optimizers", package = "gips")} or its
\href{https://przechoj.github.io/gips/articles/Optimizers.html}{pkgdown page} -
A place to learn more about
the available optimizers.
\item \code{vignette("Theory", package = "gips")} or its
\href{https://przechoj.github.io/gips/articles/Theory.html}{pkgdown page} -
A place to learn more about
the math behind the \code{gips} package.
}
}
