/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Dmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5Iprivate.h"  
#include "H5VLprivate.h" 

#ifndef H5_NO_DEPRECATED_SYMBOLS

hid_t
H5Dcreate1(hid_t loc_id, const char *name, hid_t type_id, hid_t space_id, hid_t dcpl_id)
{
    void             *dset    = NULL; 
    H5VL_object_t    *vol_obj = NULL; 
    H5VL_loc_params_t loc_params;
    hid_t             ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "name parameter cannot be NULL");
    if (!*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "name parameter cannot be an empty string");

    
    if (H5CX_set_loc(loc_id) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, H5I_INVALID_HID, "can't set collective metadata read");

    if (H5P_DEFAULT == dcpl_id)
        dcpl_id = H5P_DATASET_CREATE_DEFAULT;
    else if (true != H5P_isa_class(dcpl_id, H5P_DATASET_CREATE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not dataset create property list ID");

    
    H5CX_set_dcpl(dcpl_id);

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid location identifier");

    
    if (NULL == (dset = H5VL_dataset_create(vol_obj, &loc_params, name, H5P_LINK_CREATE_DEFAULT, type_id,
                                            space_id, dcpl_id, H5P_DATASET_ACCESS_DEFAULT,
                                            H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, H5I_INVALID_HID, "unable to create dataset");

    
    if ((ret_value = H5VL_register(H5I_DATASET, dset, H5VL_OBJ_CONNECTOR(vol_obj), true)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register dataset");

done:
    if (H5I_INVALID_HID == ret_value)
        if (dset && H5VL_dataset_close(vol_obj, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, H5I_INVALID_HID, "unable to release dataset");

    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Dopen1(hid_t loc_id, const char *name)
{
    void             *dset    = NULL; 
    H5VL_object_t    *vol_obj = NULL; 
    H5VL_loc_params_t loc_params;
    hid_t             ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "name parameter cannot be NULL");
    if (!*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "name parameter cannot be an empty string");

    
    loc_params.type     = H5VL_OBJECT_BY_SELF;
    loc_params.obj_type = H5I_get_type(loc_id);

    
    if (NULL == (vol_obj = H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid location identifier");

    
    if (NULL == (dset = H5VL_dataset_open(vol_obj, &loc_params, name, H5P_DATASET_ACCESS_DEFAULT,
                                          H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTOPENOBJ, H5I_INVALID_HID, "unable to open dataset");

    
    if ((ret_value = H5VL_register(H5I_DATASET, dset, H5VL_OBJ_CONNECTOR(vol_obj), true)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTREGISTER, H5I_INVALID_HID, "can't register dataset ID");

done:
    if (H5I_INVALID_HID == ret_value)
        if (dset && H5VL_dataset_close(vol_obj, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, H5I_INVALID_HID, "unable to release dataset");

    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Dextend(hid_t dset_id, const hsize_t size[])
{
    H5VL_object_t               *vol_obj;                 
    H5VL_dataset_get_args_t      vol_get_cb_args;         
    H5VL_dataset_specific_args_t vol_spec_cb_args;        
    hid_t                        sid = H5I_INVALID_HID;   
    H5S_t                       *ds  = NULL;              
    int                          ndims;                   
    hsize_t                      dset_dims[H5S_MAX_RANK]; 
    int                          i;                       
    herr_t                       ret_value = SUCCEED;     

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(dset_id, H5I_DATASET)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid dataset identifier");
    if (!size)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no size specified");

    
    vol_get_cb_args.op_type                 = H5VL_DATASET_GET_SPACE;
    vol_get_cb_args.args.get_space.space_id = H5I_INVALID_HID;

    
    if (H5VL_dataset_get(vol_obj, &vol_get_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get dataspace");
    sid = vol_get_cb_args.args.get_space.space_id;
    if (H5I_INVALID_HID == sid)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "received an invalid dataspace from the dataset");
    if (NULL == (ds = (H5S_t *)H5I_object_verify(sid, H5I_DATASPACE)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "couldn't get dataspace structure from ID");

    
    if (H5S_get_simple_extent_dims(ds, dset_dims, NULL) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get dataset dimensions");

    
    ndims = H5S_GET_EXTENT_NDIMS(ds);

    
    for (i = 0; i < ndims; i++)
        if (size[i] > dset_dims[i])
            dset_dims[i] = size[i];

    
    if (H5CX_set_loc(dset_id) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set collective metadata read info");

    
    vol_spec_cb_args.op_type              = H5VL_DATASET_SET_EXTENT;
    vol_spec_cb_args.args.set_extent.size = dset_dims;

    
    if (H5VL_dataset_specific(vol_obj, &vol_spec_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to extend dataset");

done:
    
    if (sid != H5I_INVALID_HID && H5I_dec_app_ref(sid) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "can't close dataspace");

    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Dvlen_reclaim(hid_t type_id, hid_t space_id, hid_t dxpl_id, void *buf)
{
    const H5T_t *type;
    H5S_t       *space;     
    herr_t       ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (buf == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "'buf' pointer is NULL");
    if (NULL == (type = (const H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid datatype");
    if (NULL == (space = (H5S_t *)H5I_object_verify(space_id, H5I_DATASPACE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid dataspace");
    if (!(H5S_has_extent(space)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "dataspace does not have extent set");

    
    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not xfer parms");

    
    H5CX_set_dxpl(dxpl_id);

    
    ret_value = H5T_reclaim(type, space, buf);

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Dread_chunk1(hid_t dset_id, hid_t dxpl_id, const hsize_t *offset, uint32_t *filters ,
               void *buf )
{
    H5VL_object_t                      *vol_obj;             
    H5VL_optional_args_t                vol_cb_args;         
    H5VL_native_dataset_optional_args_t dset_opt_args;       
    herr_t                              ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object_verify(dset_id, H5I_DATASET)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "dset_id is not a dataset ID");
    if (!buf)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "buf cannot be NULL");
    if (!offset)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "offset cannot be NULL");
    if (!filters)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "filters cannot be NULL");

    
    if (H5P_DEFAULT == dxpl_id)
        dxpl_id = H5P_DATASET_XFER_DEFAULT;
    else if (true != H5P_isa_class(dxpl_id, H5P_DATASET_XFER))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "dxpl_id is not a dataset transfer property list ID");

    
    dset_opt_args.chunk_read.offset   = offset;
    dset_opt_args.chunk_read.filters  = 0;
    dset_opt_args.chunk_read.buf      = buf;
    dset_opt_args.chunk_read.buf_size = NULL;
    vol_cb_args.op_type               = H5VL_NATIVE_DATASET_CHUNK_READ;
    vol_cb_args.args                  = &dset_opt_args;

    
    if (H5VL_dataset_optional(vol_obj, &vol_cb_args, dxpl_id, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "can't read unprocessed chunk data");

    
    *filters = dset_opt_args.chunk_read.filters;

done:
    FUNC_LEAVE_API(ret_value)
} 

#endif 
