#include <Rcpp.h>
using namespace Rcpp;

namespace {

int wsampler(NumericVector w) {
    int l = 0, r = w.size() - 1, m;
    double v = R::runif(0, w[r]);
    while (l < r) {
        if (v < w[l]) return l;
        if (v > w[r - 1]) return r;
        m = std::floor((l + r) / 2);
        if (v < w[m])
            r = m;
        else
            l = m + 1;
    }
    return l;
}

}  //  namespace

//' @name trait-model
//' @param theta a vector of length 4 containing the model parameters,
//' @param nspecies the number of different levels of the trait
//'        characteristic (default 1000)
//' @param population number of individuals living in the community
//' (default 500)
//' @param ngen number of generations (death/birth cycles) after which the
//'             actual population is returned (default 5000)
//' @returns
//' `traitSim` returns a integer vector of length `nspecies` containing the trait distribution 
//' of `population` individuals after `ngen` generations 
//' @export
// [[Rcpp::export]]
IntegerVector traitSim(NumericVector theta, int nspecies = 1000,
                       int population = 500, int ngen = 5000) {
    int r, l, i;
    double u, v, tau = theta[0], mu = theta[1], sigma = theta[2],
                 gamma = theta[3];
    IntegerVector n(nspecies);
    NumericVector fitness(nspecies), region(nspecies), local(nspecies),
        localfit(nspecies);
    // 1 Fitness and regional distribution
    for (r = 0, u = 0, v = nspecies - 1.0; r < nspecies; r++) {
        fitness[r] = 1 - gamma + gamma * R::dnorm(r / v, mu, sigma, 0);
        u += fitness[r];
        region[r] = u;
    }
    //  2. Initial local population
    for (r = 0; r < population; r++) n[wsampler(region)] += 1;
    for (r = 0, u = 0, v = 0; r < nspecies; r++) {
        u += n[r];
        local[r] = u;
        v += n[r] * fitness[r];
        localfit[r] = v;
    }
    // 3. Simulation of population after ngen generations
    for (l = 0; l < ngen; l++) {
        // "Departing" individual
        i = wsampler(local);
        n[i] -= 1;
        for (r = i, u = fitness[i]; r < nspecies; r++) {
            local[r] -= 1;
            localfit[r] -= u;
        }
        // New individuals
        i = wsampler((R::runif(0, 1) < tau) ? region : localfit);
        n[i] += 1;
        for (r = i, u = fitness[i]; r < nspecies; r++) {
            local[r] += 1;
            localfit[r] += u;
        }
    }
    return n;
}
