% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/est_score.R
\name{est_score}
\alias{est_score}
\alias{est_score.default}
\alias{est_score.est_irt}
\title{Estimate examinees' ability (proficiency) parameters}
\usage{
est_score(x, ...)

\method{est_score}{default}(
  x,
  data,
  D = 1,
  method = "ML",
  range = c(-5, 5),
  norm.prior = c(0, 1),
  nquad = 41,
  weights = NULL,
  fence.a = 3,
  fence.b = NULL,
  tol = 1e-04,
  max.iter = 100,
  se = TRUE,
  stval.opt = 1,
  intpol = TRUE,
  range.tcc = c(-7, 7),
  missing = NA,
  ncore = 1,
  ...
)

\method{est_score}{est_irt}(
  x,
  method = "ML",
  range = c(-5, 5),
  norm.prior = c(0, 1),
  nquad = 41,
  weights = NULL,
  fence.a = 3,
  fence.b = NULL,
  tol = 1e-04,
  max.iter = 100,
  se = TRUE,
  stval.opt = 1,
  intpol = TRUE,
  range.tcc = c(-7, 7),
  missing = NA,
  ncore = 1,
  ...
)
}
\arguments{
\item{x}{A data frame containing item metadata (e.g., item parameters, number
of categories, IRT model types, etc.); or an object of class \code{est_irt}
obtained from \code{\link[=est_irt]{est_irt()}}, or \code{est_item} from \code{\link[=est_item]{est_item()}}.

See \code{\link[=est_irt]{est_irt()}} or \code{\link[=simdat]{simdat()}} for more details about the item
metadata. This data frame can be easily created using the
\code{\link[=shape_df]{shape_df()}} function.}

\item{...}{Additional arguments passed to \code{\link[parallel:makeCluster]{parallel::makeCluster()}}.}

\item{data}{A matrix of examinees' item responses corresponding to the items
specified in the \code{x} argument. Rows represent examinees and columns
represent items.}

\item{D}{A scaling constant used in IRT models to make the logistic function
closely approximate the normal ogive function. A value of 1.7 is commonly
used for this purpose. Default is 1.}

\item{method}{A character string indicating the scoring method to use.
Available options are:
\itemize{
\item \code{"ML"}: Maximum likelihood estimation
\item \code{"MLF"}: Maximum likelihood estimation with fences (Han, 2016)
\item \code{"WL"}: Weighted likelihood estimation (Warm, 1989)
\item \code{"MAP"}: Maximum a posteriori estimation (Hambleton et al., 1991)
\item \code{"EAP"}: Expected a posteriori estimation (Bock & Mislevy, 1982)
\item \code{"EAP.SUM"}: Expected a posteriori summed scoring (Thissen et al., 1995;
Thissen & Orlando, 2001)
\item \code{"INV.TCC"}: Inverse test characteristic curve scoring
(e.g., Kolen & Brennan, 2004; Kolen & Tong, 2010; Stocking, 1996)
}

Default is \code{"ML"}.}

\item{range}{A numeric vector of length two specifying the lower and upper
bounds of the ability scale. This is used for the following scoring
methods: \code{"ML"}, \code{"MLF"}, \code{"WL"}, and \code{"MAP"}. Default is \code{c(-5, 5)}.}

\item{norm.prior}{A numeric vector of length two specifying the mean and
standard deviation of the normal prior distribution. These values are used
to generate the Gaussian quadrature points and weights. Ignored if \code{method}
is \code{"ML"}, \code{"MLF"}, \code{"WL"}, or \code{"INV.TCC"}. Default is \code{c(0, 1)}.}

\item{nquad}{An integer indicating the number of Gaussian quadrature points
to be generated from the normal prior distribution. Used only when \code{method}
is \code{"EAP"} or \code{"EAP.SUM"}. Ignored for \code{"ML"}, \code{"MLF"}, \code{"WL"}, \code{"MAP"},
and \code{"INV.TCC"}. Default is \code{41}.}

\item{weights}{A two-column matrix or data frame containing the quadrature
points (in the first column) and their corresponding weights (in the second
column) for the latent variable prior distribution. The weights and points
can be conveniently generated using the function \code{\link[=gen.weight]{gen.weight()}}.

If \code{NULL} and \code{method} is either \code{"EAP"} or \code{"EAP.SUM"}, default quadrature
values are generated based on the \code{norm.prior} and \code{nquad} arguments.
Ignored if \code{method} is \code{"ML"}, \code{"MLF"}, \code{"WL"}, \code{"MAP"}, or \code{"INV.TCC"}.}

\item{fence.a}{A numeric value specifying the item slope parameter (i.e.,
\emph{a}-parameter) for the two imaginary items used in MLF. See \strong{Details} below.
Default is 3.0.}

\item{fence.b}{A numeric vector of length two specifying the lower and upper
bounds of the item difficulty parameters (i.e., \emph{b}-parameters) for the two
imaginary items in MLF. If \code{fence.b = NULL}, the values specified in the
\code{range} argument are used instead. Default is NULL.}

\item{tol}{A numeric value specifying the convergence tolerance for the ML,
MLF, WL, MAP, and inverse TCC scoring methods. Newton-Raphson optimization
is used for ML, MLF, WL, and MAP, while the bisection method is used for
inverse TCC. Default is 1e-4.}

\item{max.iter}{A positive integer specifying the maximum number of
iterations allowed for the Newton-Raphson optimization. Default is 100.}

\item{se}{Logical. If \code{TRUE}, standard errors of ability estimates are
computed. If \code{method} is "EAP.SUM" or "INV.TCC", standard errors are always
returned regardless of this setting. Default is \code{TRUE}.}

\item{stval.opt}{A positive integer specifying the starting value option for
the ML, MLF, WL, and MAP scoring methods. Available options are:
\itemize{
\item 1: Brute-force search (default)
\item 2: Based on observed sum scores
\item 3: Fixed at 0
}

See \strong{Details} below for more information.}

\item{intpol}{Logical. If \code{TRUE} and \code{method = "INV.TCC"}, linear
interpolation is applied to approximate ability estimates for sum scores
that cannot be directly mapped using the TCC (e.g., when the observed sum
score is less than the total of item guessing parameters). Default is
\code{TRUE}. See \strong{Details} below.}

\item{range.tcc}{A numeric vector of length two specifying the lower and
upper bounds of ability estimates when \code{method = "INV.TCC"}. Default is
\code{c(-7, 7)}.}

\item{missing}{A value indicating missing responses in the data set. Default
is \code{NA}. See \strong{Details} below.}

\item{ncore}{An integer specifying the number of logical CPU cores to use for
parallel processing. Default is 1. See \strong{Details} below.}
}
\value{
When \code{method} is one of \code{"ML"}, \code{"MLF"}, \code{"WL"}, \code{"MAP"}, or \code{"EAP"},
a two-column data frame is returned:
\itemize{
\item Column 1: Ability estimates
\item Column 2: Standard errors of the ability estimates
}

When \code{method} is either \code{"EAP.SUM"} or \code{"INV.TCC"}, a list with two
components is returned:
\itemize{
\item Object 1: A three-column data frame including:
\itemize{
\item Column 1: Observed sum scores
\item Column 2: Ability estimates
\item Column 3: Standard errors of the ability estimates
}
\item Object 2: A score table showing possible raw sum scores and the corresponding
ability and standard error estimates
}
}
\description{
This function estimates examinees' latent ability parameters. Available
scoring methods include maximum likelihood estimation (ML), maximum
likelihood estimation with fences (MLF; Han, 2016), weighted likelihood
estimation (WL; Warm, 1989), maximum a posteriori estimation (MAP; Hambleton
et al., 1991), expected a posteriori estimation (EAP; Bock & Mislevy, 1982),
EAP summed scoring (Thissen et al., 1995; Thissen & Orlando, 2001), and
inverse test characteristic curve (TCC) scoring (e.g., Kolen & Brennan, 2004;
Kolen & Tong, 2010; Stocking, 1996).
}
\details{
For the MAP scoring method, only a normal prior distribution is
supported for the population distribution.

When there are missing responses in the data set, the missing value must be
explicitly specified using the \code{missing} argument. Missing data are
properly handled when using the ML, MLF, WL, MAP, or EAP methods. However,
when using the "EAP.SUM" or "INV.TCC" methods, any missing responses are
automatically treated as incorrect (i.e., recoded as 0s).

In the maximum likelihood estimation with fences (MLF; Han, 2016), two
imaginary items based on the 2PL model are introduced. The first imaginary
item functions as the lower fence, and its difficulty parameter (\emph{b})
should be smaller than any of the difficulty parameters in the test form.
Similarly, the second imaginary item serves as the upper fence, and its \emph{b}
parameter should be greater than any difficulty value in the test form.
Both imaginary items should also have very steep slopes (i.e., high
\emph{a}-parameter values). See Han (2016) for more details. If \code{fence.b = NULL}, the function will automatically assign the lower and upper fences
based on the values provided in the \code{range} argument.

When the "INV.TCC" method is used with the 3PL model, ability estimates
cannot be obtained for observed sum scores that are less than the sum of
the items' guessing parameters. In such cases, linear interpolation can be
applied by setting \code{intpol = TRUE}.

Let \eqn{\theta_{min}} and \eqn{\theta_{max}} denote the minimum and
maximum ability estimates, respectively, and let \eqn{\theta_{X}} be the
ability estimate corresponding to the smallest observed sum score, X, that
is greater than or equal to the sum of the guessing parameters.When linear
interpolation is applied, the first value in the \code{range.tcc} argument is
treated as \eqn{\theta_{min}}. A line is then constructed between the
points \eqn{(x = \theta_{min}, y = 0)} and \eqn{(x = \theta_{X}, y = X)}.
The second value in \code{range.tcc} is interpreted as \eqn{\theta_{max}}, which
corresponds to the ability estimate for the maximum observed sum score.

For the "INV.TCC" method, standard errors of ability estimates are computed
using the approach proposed by Lim et al. (2020). The implementation of
inverse TCC scoring in this function is based on a modified version of the
\code{SNSequate::irt.eq.tse()} function from the \pkg{SNSequate} package
(González, 2014).

For the ML, MLF, WL, and MAP scoring methods, different strategies can be
used to determine the starting value for ability estimation based on the
\code{stval.opt} argument:
\itemize{
\item When \code{stval.opt = 1} (default), a brute-force search is performed by
evaluating the log-likelihood at discrete theta values within the range
specified by \code{range}, using 0.1 increments. The theta value yielding the
highest log-likelihood is chosen as the starting value.
\item When \code{stval.opt = 2}, the starting value is derived from the observed
sum score using a logistic transformation. For example, if the maximum
possible score (\code{max.score}) is 30 and the examinee’s observed sum score
(\code{obs.score}) is 20, the starting value is \code{log(obs.score / (max.score -   obs.score))}.
\itemize{
\item If all responses are incorrect (i.e., \code{obs.score = 0}), the starting
value is \code{log(1 / max.score)}.
\item If all responses are correct (\code{obs.score = max.score}), the starting
value is \code{log(max.score / 1)}.
}
\item When \code{stval.opt = 3}, the starting value is fixed at 0.
}

To accelerate ability estimation using the ML, MLF, WL, MAP, and EAP
methods, this function supports parallel processing across multiple logical
CPU cores. The number of cores can be specified via the \code{ncore} argument
(default is 1).

Note that the standard errors of ability estimates are computed based on
the Fisher expected information for the ML, MLF, WL, and MAP methods.

For the implementation of the WL method, the function references the
\code{catR::Pi()}, \code{catR::Ji()}, and \code{catR::Ii()} functions from the \pkg{catR}
package (Magis & Barrada, 2017).
}
\section{Methods (by class)}{
\itemize{
\item \code{est_score(default)}: Default method to estimate examinees' latent ability
parameters using a data frame \code{x} containing the item metadata.

\item \code{est_score(est_irt)}: An object created by the function \code{\link[=est_irt]{est_irt()}}.

}}
\examples{
## Import the "-prm.txt" output file from flexMIRT
flex_prm <- system.file("extdata", "flexmirt_sample-prm.txt", package = "irtQ")

# Read item parameters and convert them into item metadata
x <- bring.flexmirt(file = flex_prm, "par")$Group1$full_df

# Generate examinee ability values
set.seed(12)
theta <- rnorm(10)

# Simulate item response data based on the item metadata and abilities
data <- simdat(x, theta, D = 1)

\donttest{
# Estimate abilities using maximum likelihood (ML)
est_score(x, data, D = 1, method = "ML", range = c(-4, 4), se = TRUE)

# Estimate abilities using weighted likelihood (WL)
est_score(x, data, D = 1, method = "WL", range = c(-4, 4), se = TRUE)

# Estimate abilities using MLF with default fences
# based on the `range` argument
est_score(x, data, D = 1, method = "MLF",
  fence.a = 3.0, fence.b = NULL, se = TRUE)

# Estimate abilities using MLF with user-specified fences
est_score(x, data, D = 1, method = "MLF", fence.a = 3.0,
  fence.b = c(-7, 7), se = TRUE)

# Estimate abilities using maximum a posteriori (MAP)
est_score(x, data, D = 1, method = "MAP", norm.prior = c(0, 1),
  nquad = 30, se = TRUE)

# Estimate abilities using expected a posteriori (EAP)
est_score(x, data, D = 1, method = "EAP", norm.prior = c(0, 1),
  nquad = 30, se = TRUE)

# Estimate abilities using EAP summed scoring
est_score(x, data, D = 1, method = "EAP.SUM", norm.prior = c(0, 1),
  nquad = 30)

# Estimate abilities using inverse TCC scoring
est_score(x, data, D = 1, method = "INV.TCC", intpol = TRUE,
  range.tcc = c(-7, 7))
}

}
\references{
Bock, R. D., & Mislevy, R. J. (1982). Adaptive EAP estimation of
ability in a microcomputer environment. \emph{Psychometrika, 35}, 179-198.

González, J. (2014). SNSequate: Standard and nonstandard statistical models
and methods for test equating.
\emph{Journal of Statistical Software, 59}, 1-30.

Hambleton, R. K., Swaminathan, H., & Rogers, H. J. (1991).\emph{Fundamentals of
item response theory}. Newbury Park, CA: Sage.

Han, K. T. (2016). Maximum likelihood score estimation method with fences
for short-length tests and computerized adaptive tests.
\emph{Applied psychological measurement, 40}(4), 289-301.

Howard, J. P. (2017). \emph{Computational methods for numerical analysis with
R}. New York: Chapman and Hall/CRC.

Kolen, M. J. & Brennan, R. L. (2004). \emph{Test Equating, Scaling, and Linking}
(2nd ed.). New York: Springer

Kolen, M. J. & Tong, Y. (2010). Psychometric properties of IRT proficiency
estimates.
\emph{Educational Measurement: Issues and Practice, 29}(3), 8-14.

Lim, H., Davey, T., & Wells, C. S. (2020). A recursion-based analytical
approach to evaluate the performance of MST.
\emph{Journal of Educational Measurement, 58}(2), 154-178.

Magis, D., & Barrada, J. R. (2017). Computerized adaptive testing with R:
Recent updates of the package catR.
\emph{Journal of Statistical Software, 76}, 1-19.

Stocking, M. L. (1996). An alternative method for scoring adaptive tests.
\emph{Journal of Educational and Behavioral Statistics, 21}(4), 365-389.

Thissen, D. & Orlando, M. (2001). Item response theory for items scored in
two categories. In D. Thissen & H. Wainer (Eds.),
\emph{Test scoring} (pp.73-140). Mahwah, NJ: Lawrence Erlbaum.

Thissen, D., Pommerich, M., Billeaud, K., & Williams, V. S. (1995). Item
Response Theory for Scores on Tests Including Polytomous Items with Ordered
Responses. \emph{Applied Psychological Measurement, 19}(1), 39-49.

Warm, T. A. (1989). Weighted likelihood estimation of ability in item
response theory. \emph{Psychometrika, 54}(3), 427-450.
}
\seealso{
\code{\link[=est_irt]{est_irt()}}, \code{\link[=simdat]{simdat()}}, \code{\link[=shape_df]{shape_df()}},
\code{\link[=gen.weight]{gen.weight()}}
}
\author{
Hwanggyu Lim \email{hglim83@gmail.com}
}
