% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/2_iteror.R, R/iter.connection.R
\name{iteror}
\alias{iteror}
\alias{iteror.iter}
\alias{iteror.default}
\alias{iteror.connection}
\title{Make an iteror from a given object.}
\usage{
iteror(obj, ...)

\method{iteror}{iter}(obj, ...)

\method{iteror}{default}(obj, ..., recycle = FALSE, chunkSize, chunks)

\method{iteror}{connection}(obj, ...)
}
\arguments{
\item{obj}{An object to iterate with.}

\item{...}{Different \code{iteror} methods may take additional options
depending on the class of \code{obj}.}

\item{recycle}{a boolean describing whether the iterator should
reset after running through all its values.}

\item{chunkSize}{How many elements (or slices) to include in each
chunk.}

\item{chunks}{Split the input into this many chunks.}
}
\value{
an object of classes 'iteror' and 'iter'.

The method \code{iteror.iter} wraps an \link[iterators:iter]{iterators::iter} object
and returns an iteror.

The default method \code{iteror.default} treats \code{obj} as a
vector to yield values from.
}
\description{
\code{it <- iteror(obj, ...)} is a generic constructor that creates
objects of class "iteror" from its input. An iteror outputs a
single element of a sequence each time you call
\code{nextOr(it)}. Different \code{iteror} methods exist for different data
types and may take different optional arguments as listed in this
page.
}
\details{
When called, an iteror may either return a new value or stop. The
way an iteror signals a stop is that it does whatever you write in
the argument \code{or}. For instance you can write \code{or=break} to exit a
loop. Summing over an iteror this way looks like:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sum <- 0
it <- iteror(iseq(0, 100, 7))
repeat \{
  sum <- sum + nextOr(it, break)
\}
}\if{html}{\out{</div>}}

Another way to use the "or" argument is to give it a sentinel value;
that is, a special value that you will interpret as end of
iteration.  If the result of calling \code{nextOr} is \code{identical()} to
the value you provided, then you know the iterator has ended. This
pattern looks like:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sum <- 0
stopped <- new.env()
it <- iteror(iseq(0, 100, 7))
repeat \{
  val <- nextOr(it, stopped)
  if (identical(val, stopped)) break
  sum <- sum + val
\}
}\if{html}{\out{</div>}}

(Here I'm using \code{new.env()} as a sentinel value. In R it is
commonplace to use \code{NULL} or \code{NA} as a kind of sentinel value, but
that only works until you have an iterator that needs to yield NULL
itself. A safer alternative is to use a local, one-shot sentinel value;
\code{new.env()} is ideal, as it constructs an object that is
not \link{identical} to any other object in the R session.)

Note that \code{iteror} objects are simply functions with a class
attribute attached, and all \code{nextOr.iteror} does is call the
function. So if you were in the mood, you could skip calling
\code{nextOr} involving S3 dispatch and instead call the iteror
directly. If you take this approach, make sure you have called
\code{iteror()} first to ensure that you have a true \code{iteror} object.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sum <- 0
it <- iteror(iseq(0, 100, 7))
repeat sum <- sum + it(or=break)
sum
#> [1] 735
}\if{html}{\out{</div>}}

To create iterors with custom-defined behavior, see
\link{iteror.function}.
}
\seealso{
iteror.array iteror.function iteror.data.frame
}
