#' The two-step algorithm to calculate the best k value for the improved trimmed Hochberg method to ensure that the maximum type I error rate reaches alpha exactly when rho is arbitrary
#'
#' @param alpha the significance level
#' @param alphavec a numeric vector of two values representing the weighted significance levels assigned to the two hypotheses
#'
#' @returns the best k value `k_opt` and the rho value that makes the type I error rate reaches the maximum value `rho_opt`
#' @export
#'
#' @importFrom stats uniroot
#' @author Jiangtao Gou
#' @author Fengqing Zhang
#' @references
#' Gou, J., Chang, Y., Li, T., and Zhang, F. (2025). Improved trimmed weighted Hochberg procedures with two endpoints and sample size optimization. Technical Report.
#' @examples
#' optk(alpha = 0.025)
optk <- function (alpha, alphavec = c(alpha/2, alpha/2)) {
  # R20241108e
  # Rfun_optk
  find_k_result <- stats::uniroot(f = find_k_target_mvtnorm, interval = c(0, 1), tol = .Machine$double.eps^0.8, alpha = alpha, alphavec = alphavec)
  k_opt <- find_k_result$root
  #
  find_rho_result <- stats::uniroot(f = find_rho_target_mvtnorm, interval = c(0, 0.5), tol = .Machine$double.eps^0.8, k = k_opt, alpha = alpha, alphavec = alphavec)
  rho_opt <- find_rho_result$root
  #
  return(list(k_opt = k_opt, rho_opt = rho_opt))
}
