% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/01_generics.R
\name{loglikelihood}
\alias{loglikelihood}
\alias{ll}
\title{Log-Likelihood Function}
\usage{
ll(distr, x, ...)
}
\arguments{
\item{distr}{A \code{Distribution} object}

\item{x}{numeric. A sample under estimation.}

\item{...}{extra arguments.}
}
\value{
If only the \code{distr} argument is supplied, \code{ll()} returns a function.
If both \code{distr} and \code{x} are supplied, \code{ll()} returns a numeric, the value of
the log-likelihood function.
}
\description{
This function calculates the log-likelihood of an independent and
identically distributed (iid) sample from a distribution. See Details.
}
\details{
The log-likelihood functions are provided in two forms: the \verb{ll<name>}
distribution-specific version that follows the stats package conventions, and
the S4 generic \code{ll}. Examples for the \verb{ll<name>} version are included in the
distribution-specific help pages, e.g. \code{?Beta} (all distributions can be
found in the See Also section of the \code{Distributions} help page).

As with the \code{d()}, \code{p()}, \code{q()}, \code{r()} methods, \code{ll()} can be supplied only
with \code{distr} to return the log-likelihood function (i.e. it can be used as a
functional), or with both \code{distr} and \code{x} to be evaluated directly.

In some distribution families like beta and gamma, the MLE cannot be
explicitly derived and numerical optimization algorithms have to be employed.
Even in ``good" scenarios, with plenty of observations and a smooth
optimization function, extra care should be taken to ensure a fast and right
convergence if possible. Two important steps are taken in package in this
direction:
\enumerate{
\item The log-likelihood function is analytically calculated for each
distribution family, so that constant terms with respect to the parameters
can be removed, leaving only the sufficient statistics as a requirement for
the function evaluation.
\item Multidimensional problems are reduced to unidimensional ones by utilizing
the score equations.
}

The resulting function that is inserted in the optimization algorithm is
called \code{lloptim()}, and is not to be confused with the actual log-likelihood
function \code{ll()}. The corresponding derivative is called \code{dlloptim()}. These
functions are used internally and are not exported.

Therefore, whenever numerical computation of the MLE is required, \code{optim()}
is called to optimize \code{lloptim()}, using the ME or SAME as the starting point
(user's choice), and the L-BFGS-U optimization algorithm, with lower and
upper limits defined by default as the parameter space boundary. Illustrative
examples can be found in the package vignette.
}
\examples{
# -----------------------------------------------------
# Beta Distribution Example
# -----------------------------------------------------

# Create the distribution
a <- 3
b <- 5
D <- Beta(a, b)

# ------------------
# dpqr Functions
# ------------------

d(D, c(0.3, 0.8, 0.5)) # density function
p(D, c(0.3, 0.8, 0.5)) # distribution function
qn(D, c(0.4, 0.8)) # inverse distribution function
x <- r(D, 100) # random generator function

# alternative way to use the function
df <- d(D) ; df(x) # df is a function itself

# ------------------
# Moments
# ------------------

mean(D) # Expectation
var(D) # Variance
sd(D) # Standard Deviation
skew(D) # Skewness
kurt(D) # Excess Kurtosis
entro(D) # Entropy
finf(D) # Fisher Information Matrix

# List of all available moments
mom <- moments(D)
mom$mean # expectation

# ------------------
# Point Estimation
# ------------------

ll(D, x)
llbeta(x, a, b)

ebeta(x, type = "mle")
ebeta(x, type = "me")
ebeta(x, type = "same")

mle(D, x)
me(D, x)
same(D, x)
e(D, x, type = "mle")

mle("beta", x) # the distr argument can be a character

# ------------------
# Estimator Variance
# ------------------

vbeta(a, b, type = "mle")
vbeta(a, b, type = "me")
vbeta(a, b, type = "same")

avar_mle(D)
avar_me(D)
avar_same(D)

v(D, type = "mle")
}
\seealso{
\link{distributions}, \link{moments}, \link{estimation}
}
