% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LazyBar.R
\name{lazyProgressBar}
\alias{lazyProgressBar}
\title{Progress bar with customisable estimated remaining time}
\usage{
lazyProgressBar(n, method = "average", fn = NULL, ...)
}
\arguments{
\item{n}{Integer. Total number of ticks}

\item{method}{Character. The embedded forecasting method of remaining time:
\code{drift} (default), \code{average}, \code{naive}, or \code{snaive}.
Ignored if \code{fn} is not \code{NULL}.
\describe{
\item{\code{average} (default)}{Average method. The run time between future ticks are assumed to
be the average run time of the past ticks.
This is the most common estimation method for remaining time.}
\item{\code{drift}}{Drift method. The run time between future ticks are
assumed to increase (decrease), and the level changed is set to be the average change
of the run time of the past ticks.
This is to assume the run time between ticks is linearly increasing or decreasing.}
\item{\code{naive}}{Naive method. The run time between future ticks are assumed to be
the run time between the last two ticks,}
\item{\code{snaive}}{Seasonal naive method. If this method is chosen, an argument of \code{s}
needs to be supplied in the \code{...}.
The run time between future ticks is set to be the run time \code{s} times before.
By default \code{s} is set to be 1/10 of the total number of ticks.}
}}

\item{fn}{Function. User defined function to estimate the remaining time.
The function should predict the remaining time using the arguments and
return a scalar.
It should have at least three arguments in the order of \code{dtime}, \code{i}, and \code{n},
which represent the status of the progress bar at the current tick:
\describe{
\item{\code{dtime}}{A numeric vector of the run time between past ticks.}
\item{\code{i}}{The number of the current tick.}
\item{\code{n}}{The number of total ticks.}
}}

\item{...}{Other arguments to pass to estimation method.
The arguments need to be named.}
}
\value{
An R6 object with methods \code{tick()} and \code{print()}.
}
\description{
Display a progress bar displaying the estimated time.
The purpose of having various estimation methods is to
provide a more accurate estimation when the run time between
ticks is assumed to be different, e.g., online estimation,
time series cross validation, expanding window approach, etc.
}
\details{
Four simple forecasting methods are available for
the estimation of the remaining time:
Average method (default), Drift method, Naive method and
Seasonal naive method.
For the summary of the simple methods, see Chapter 3 of \code{References}.
User can also supply their customised estimation method as a function.
See \code{Arguments} and \code{Examples}.
}
\examples{
\donttest{
pb <- lazyProgressBar(4)
pb$tick()
pb$tick()
pb$tick()
pb$tick()

# With linearly increasing run time
pb <- lazyProgressBar(4, method = "drift")
for(i in 1:4){
  Sys.sleep(i * 0.2)
  pb$tick()$print()
}

# With user defined forecast function
# The forecast function itself will
# require certain computational power
forecast_fn <- function(dtime, i, n, s = 10){
  # When the number of ticks is smaller than s
  # Estimate the future run time
  # as the average of the past
  if(i<s){
    eta <- mean(dtime)*(n-i)
  }

  # When the number of ticks is larger than s
  # Fit an arima model every s ticks
  # using forecast package
  if(i>=s){
    if(i \%\% s ==0){
      model <- forecast::auto.arima(dtime)
    }
    runtime <- forecast::forecast(model, h=n-i)$mean
    if(i \%\% s !=0){
      runtime <- runtime[-seq_len(i \%\% s)]
    }
    eta <- sum(runtime)
  }
  return(eta)
}

pb <- lazyProgressBar(10, fn = forecast_fn, s=3)
for(i in 1:10){
  Sys.sleep(i * 0.2)
  pb$tick()$print()
}
}
}
\references{
Hyndman, R.J., & Athanasopoulos, G. (2018) Forecasting: principles and practice, 2nd edition, OTexts: Melbourne, Australia. OTexts.com/fpp2. Accessed on 24/04/2020.
}
\author{
Yangzhuoran Fin Yang
}
