% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lingmatch.R
\name{lingmatch}
\alias{lingmatch}
\title{Linguistic Matching and Accommodation}
\usage{
lingmatch(input = NULL, comp = mean, data = NULL, group = NULL, ...,
  comp.data = NULL, comp.group = NULL, order = NULL, drop = FALSE,
  all.levels = FALSE, type = "lsm")
}
\arguments{
\item{input}{Texts to be compared; a vector, document-term matrix (dtm; with terms as column names),
or path to a file (.txt or .csv, with texts separated by one or more lines/rows).}

\item{comp}{Defines the comparison to be made:
\itemize{
  \item If a \strong{function}, this will be applied to \code{input} within each group (overall if there is
    no group; i.e., \code{apply(input, 2, comp)}; e.g., \code{comp = mean} would compare each text to
    the mean profile of its group).
  \item If a \strong{character} with a length of 1 and no spaces:
    \itemize{
      \item If it partially matches one of \code{lsm_profiles}'s rownames, that row will be used as the comparison.
      \item If it partially matches \code{'auto'}, the highest correlating \code{lsm_profiles} row will be used.
      \item If it partially matches \code{'pairwise'}, each text will be compared to one another.
      \item If it partially matches \code{'sequential'}, the last variable in \code{group} will be treated as
        a speaker ID (see the Grouping and Comparisons section).
    }
  \item If a \strong{character vector}, this will be processed in the same way as \code{input}.
  \item If a \strong{vector}, either (a) logical or factor-like (having n levels < length) and of the same length as
    \code{nrow(input)}, or (b) numeric or logical of length less than \code{nrow(input)}, this will be used to
    select a subset of \code{input} (e.g., \code{1:10} would treat the first 10 rows of \code{input} as the
    comparison; \code{lingmatch(text, type == 'prompt', data)} would use the texts in the \code{text} column
    identified by the \code{type} column as the comparison).
  \item If a \strong{matrix-like object} (having multiple rows and columns), or a named vector, this will
    be treated as a sort of dtm, assuming there are common (column) names between \code{input} and
    \code{comp} (e.g., if you had prompt and response texts that were already processed separately).
}}

\item{data}{A matrix-like object as a reference for column names, if variables are referred to in
other arguments (e.g., \code{lingmatch(text, data = data)} would be the same as
\code{lingmatch(data$text)}.}

\item{group}{A logical or factor-like vector the same length as \code{NROW(input)}, used to defined
groups.}

\item{...}{Passes arguments to \code{\link{lma_dtm}}, \code{\link{lma_weight}},
\code{\link{lma_termcat}}, and/or \code{\link{lma_lspace}} (depending on \code{input} and \code{comp}),
and \code{\link{lma_simets}}.}

\item{comp.data}{A matrix-like object as a source for \code{comp} variables.}

\item{comp.group}{The column name of the grouping variable(s) in \code{comp.data}; if
\code{group} contains references to column names, and \code{comp.group} is not specified,
\code{group} variables will be looked for in \code{comp.data}.}

\item{order}{A numeric vector the same length as \code{nrow(input)} indicating the order of the
texts and grouping variables when the type of comparison is sequential. Only necessary if the
texts are not already ordered as desired.}

\item{drop}{logical; if \code{TRUE}, will drop columns with a sum of 0.}

\item{all.levels}{logical; if \code{FALSE}, multiple groups are combined. See the Grouping and
Comparisons section.}

\item{type}{A character at least partially matching 'lsm' or 'lsa'; applies default settings
aligning with the standard calculations of each type:
\tabular{ll}{
  LSM \tab \code{lingmatch(text, weight = 'freq', dict = lma_dict(1:9), metric = 'canberra')}\cr
  LSA \tab \code{lingmatch(text, weight = 'tfidf', space = '100k_lsa', metric = 'cosine')}\cr
}}
}
\value{
A list with processed components of the input, information about the comparison, and results of
the comparison:
\itemize{
  \item \strong{\code{dtm}}: A sparse matrix; the raw count-dtm, or a version of the original input
    if it is more processed.
  \item \strong{\code{processed}}: A matrix-like object; a processed version of the input
    (e.g., weighted and categorized).
  \item \strong{\code{comp.type}}: A string describing the comparison if applicable.
  \item \strong{\code{comp}}: A vector or matrix-like object; the comparison data if applicable.
  \item \strong{\code{group}}: A string describing the group if applicable.
  \item \strong{\code{sim}}: Result of \code{\link{lma_simets}}.
}
}
\description{
Offers a variety of methods to assess linguistic matching or accommodation, where \emph{matching}
is general similarity (sometimes called \emph{homophily}), and \emph{accommodation} is some form
of conditional similarity (accounting for some base-rate or precedent; sometimes called
\emph{alignment}).
}
\details{
There are a great many points of decision in the assessment of linguistic similarity and/or
accommodation, partly inherited from the great many point of decision inherent in the numerical
representation of language. Two general types of matching are implemented here as sets of
defaults: Language/Linguistic Style Matching (LSM; Niederhoffer & Pennebaker, 2002; Ireland &
Pennebaker, 2010), and Latent Semantic Analysis/Similarity (LSA; Landauer & Dumais, 1997;
Babcock, Ta, & Ickes, 2014). See the \code{type} argument for specifics.
}
\section{Grouping and Comparisons}{

Defining groups and comparisons can sometimes be a bit complicated, and requires dataset
specific knowledge, so it can't always (readily) be done automatically. Variables entered in the
\code{group} argument are treated differently depending on their position and other arguments:

\describe{
  \item{Splitting}{By default, groups are treated as if they define separate chunks of data in
    which comparisons should be calculated. Functions used to calculated comparisons, and
    pairwise comparisons are performed separately in each of these groups. For example, if you
    wanted to compare each text with the mean of all texts in its condition, a \code{group}
    variable could identify and split by condition. Given multiple grouping variables,
    calculations will either be done in each split (if \code{all.levels = TRUE}; applied in
    sequence so that groups become smaller and smaller), or once after all splits are made (if
    \code{all.levels = FALSE}). This makes for 'one to many' comparisons with either calculated
    or preexisting standards (i.e., the profile of the current data, or a precalculated profile,
    respectively).}
  \item{Comparison ID}{When comparison data is identified in \code{comp}, groups are assumed
    to apply to both \code{input} and \code{comp} (either both in \code{data}, or separately
    between \code{data} and \code{comp.data}, in which case \code{comp.group} may be needed if
    the same grouping variable have different names between \code{data} and \code{comp.data}).
    In this case, multiple grouping variables are combined into a single factor assumed to
    uniquely identify a comparison. This makes for 'one to many' comparisons with specific texts
    (as in the case of manipulated prompts or text-based conditions).}
  \item{Speaker ID}{If \code{comp} matches \code{'sequential'}, the last grouping variable
    entered is assumed to identify something like speakers (i.e., a factor with two or more
    levels and multiple observations per level). In this case, the data are assumed to be ordered
    (or ordered once sorted by \code{order} if specified). Any additional grouping variables
    before the last are treated as splitting groups. This can set up for probabilistic
    accommodation metrics. At the moment, when sequential comparisons are made within groups,
    similarity scores between speakers are averaged, resulting in mean matching between speakers
    within the group.}
}
}

\examples{
# compare single strings
lingmatch("Compare this sentence.", "With this other sentence.")

# compare each entry in a character vector with...
texts <- c(
  "One bit of text as an entry...",
  "Maybe multiple sentences in an entry. Maybe essays or posts or a book.",
  "Could be lines or a column from a read-in file..."
)

## one another
lingmatch(texts)

## the first
lingmatch(texts, 1)

## the next
lingmatch(texts, "seq")

## the set average
lingmatch(texts, mean)

## other entries in a group
lingmatch(texts, group = c("a", "a", "b"))

## one another, without stop words
lingmatch(texts, exclude = "function")

## a standard average (based on function words)
lingmatch(texts, "auto", dict = lma_dict(1:9))

}
\references{
Babcock, M. J., Ta, V. P., & Ickes, W. (2014). Latent semantic similarity and language style
  matching in initial dyadic interactions. \emph{Journal of Language and Social Psychology, 33},
  78-88.

Ireland, M. E., & Pennebaker, J. W. (2010). Language style matching in writing: synchrony in
  essays, correspondence, and poetry. \emph{Journal of Personality and Social Psychology, 99},
  549.

Landauer, T. K., & Dumais, S. T. (1997). A solution to Plato's problem: The latent semantic
  analysis theory of acquisition, induction, and representation of knowledge.
  \emph{Psychological Review, 104}, 211.

Niederhoffer, K. G., & Pennebaker, J. W. (2002). Linguistic style matching in social interaction.
  \emph{Journal of Language and Social Psychology, 21}, 337-360.
}
\seealso{
For a general text processing function, see \code{\link{lma_process}()}.
}
