#' Creation of a log and axecution of a file
#'
#' `axecute()` creates a log, executes a file, and returns 0 if there are no
#' errors or 1 if there are any errors
#'
#' @param file String. Path to file to execute
#' @param log_name String. Name of log file
#' @param log_path String. Path to log file
#' @param include_rds Boolean. Option to export log object as Rds file.
#' Defaults to FALSE
#' @param quit_on_error Boolean. Should the session quit with status 1 on error?
#' Defaults to TRUE
#' @param to_report String vector. Objects to optionally report, may include as
#' many as necessary:
#' * messages: any messages generated by program execution
#' * output: any output generated by program execution
#' * result: any result generated by program execution
#' @param show_repo_url Boolean. Should the repository URLs be reported
#' Defaults to FALSE
#' @param extra_info List. Objects to optionally add on to end of log
#' in a special extra info section. List printed in YAML format. Optional
#' @param ... Not used
#'
#' @importFrom purrr map_chr
#'
#' @return 0 if there are no errors or 1 if there are any errors
#' @export
#'
#' @examples
#' dir <- tempdir()
#' text <- 'print("Hello, logrx-person!")'
#' fileConn <- file(file.path(dir, "hello.R"))
#' writeLines(text, fileConn)
#' close(fileConn)
#'
#' axecute(file.path(dir, "hello.R"))
#'
#'
#' fileConn <- file(file.path(dir, "hello.Rmd"))
#' writeLines(text, fileConn)
#' close(fileConn)
#'
#' axecute(file.path(dir, "hello.Rmd"))
axecute <- function(file,
                    log_name = NA,
                    log_path = NA,
                    include_rds = FALSE,
                    quit_on_error = TRUE,
                    to_report = c("messages", "output", "result"),
                    show_repo_url = FALSE,
                    extra_info = NA,
                    ...) {
  # deprecations
  if (methods::hasArg(remove_log_object)) {
    lifecycle::deprecate_stop("0.3.0", "axecute(remove_log_object = )", "axecute(include_rds = )")
  }

  # remove log object
  remove_log_object <- TRUE

  # lower everything for consistency and check values
  to_report <- map_chr(to_report, tolower)
  match.arg(to_report, several.ok = TRUE)

  # initialize log
  log_config(
    file = file,
    log_name = log_name,
    log_path = log_path,
    extra_info = extra_info
  )

  # run the code
  run_safely_loudly(file)

  # check for errors prior to log_write() since this can remove the log
  any_errors <- get_log_element("errors")

  # write log
  log_write(
    file = file,
    remove_log_object = remove_log_object,
    show_repo_url = show_repo_url,
    include_rds = include_rds,
    to_report = to_report,
    extra_info = extra_info
  )

  # if error, quit with status = 1 if not interactive
  if (!interactive() & !is.null(any_errors) & quit_on_error) {
    quit(status = 1)
  }
}
