% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/net_matrix_entropy.R
\name{net_matrix_entropy}
\alias{net_matrix_entropy}
\title{Estimate Migration Flows to Match Net Totals via Entropy Minimization}
\usage{
net_matrix_entropy(net_tot, m, zero_mask = NULL, tol = 1e-06, verbose = FALSE)
}
\arguments{
\item{net_tot}{A numeric vector of net migration totals for each region. Must sum to zero.}

\item{m}{A square numeric matrix providing prior flow estimates. Must have dimensions \code{length(net_tot) × length(net_tot)}.}

\item{zero_mask}{A logical matrix of the same dimensions as \code{m}, where \code{TRUE} indicates forbidden (structurally zero) flows. Defaults to disallowing diagonal flows.}

\item{tol}{Numeric tolerance for checking whether \code{sum(net_tot) == 0}. Default is \code{1e-6}.}

\item{verbose}{Logical flag to print solver diagnostics from \code{CVXR}. Default is \code{FALSE}.}
}
\value{
A named list with components:
\describe{
\item{\code{n}}{Estimated matrix of flows satisfying the net constraints.}
\item{\code{it}}{Number of iterations (always \code{1} for this solver).}
\item{\code{tol}}{Tolerance used for the net flow balance check.}
\item{\code{value}}{Sum of squared deviation from target net flows.}
\item{\code{convergence}}{Logical indicating successful optimization.}
\item{\code{message}}{Solver message returned by \code{CVXR}.}
}
}
\description{
Solves for an origin–destination flow matrix that satisfies directional net migration constraints
while minimizing Kullback–Leibler (KL) divergence from a prior matrix. This yields a smooth,
information-theoretically regularized solution that balances fidelity to prior patterns with net flow requirements.
}
\details{
This function minimizes the KL divergence between the estimated matrix \eqn{y_{ij}} and the prior matrix \eqn{m_{ij}}:
\deqn{\sum_{i,j} \left[y_{ij} \log\left(\frac{y_{ij}}{m_{ij}}\right) - y_{ij} + m_{ij}\right]}
subject to directional net flow constraints:
\deqn{\sum_j y_{ji} - \sum_j y_{ij} = \text{net}_i}
All flows are constrained to be non-negative. Structural zeros are enforced via \code{zero_mask}.
Internally uses \code{CVXR::kl_div()} for DCP-compliant KL minimization.
}
\examples{
m <- matrix(c(0, 100, 30, 70,
              50,   0, 45,  5,
              60,  35,  0, 40,
              20,  25, 20,  0),
            nrow = 4, byrow = TRUE,
            dimnames = list(orig = LETTERS[1:4], dest = LETTERS[1:4]))
addmargins(m)
sum_region(m)

net <- c(30, 40, -15, -55)
result <- net_matrix_entropy(net_tot = net, m = m)
result$n |>
  addmargins() |>
  round(2)
sum_region(result$n)
}
\seealso{
\code{\link{net_matrix_lp}()} for linear programming using L1 loss,
\code{\link{net_matrix_ipf}()} for iterative proportional fitting with multiplicative scaling,
and \code{\link{net_matrix_optim}()} for quadratic loss minimization.
}
