% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/net_matrix_lp.R
\name{net_matrix_lp}
\alias{net_matrix_lp}
\title{Estimate Migration Flows to Match Net Totals via Linear Programming}
\usage{
net_matrix_lp(net_tot, m, zero_mask = NULL, tol = 1e-06)
}
\arguments{
\item{net_tot}{A numeric vector of net migration totals for each region. Must sum to zero.}

\item{m}{A square numeric matrix providing prior flow estimates. Must have dimensions \code{length(net_tot) × length(net_tot)}.}

\item{zero_mask}{A logical matrix of the same dimensions as \code{m}, where \code{TRUE} indicates forbidden (structurally zero) flows. Defaults to disallowing diagonal flows.}

\item{tol}{A numeric tolerance for checking that \code{sum(net_tot) == 0}. Default is \code{1e-6}.}
}
\value{
A named list with components:
\describe{
\item{\code{n}}{Estimated matrix of flows satisfying the net constraints.}
\item{\code{it}}{Number of iterations (always \code{1} for LP method).}
\item{\code{tol}}{Tolerance used for checking net flow balance.}
\item{\code{value}}{Total L1 deviation from prior matrix \code{m}.}
\item{\code{convergence}}{Logical indicator of successful solve.}
\item{\code{message}}{Text summary of convergence status.}
}
}
\description{
Solves for an origin-destination flow matrix that satisfies directional net migration constraints
while minimizing the total absolute deviation from a prior matrix. This method uses linear
programming with split variables to minimize L1 error, optionally respecting a structural zero mask.
}
\details{
This function uses \code{lpSolve::lp()} to solve a linear program. The estimated matrix minimizes the sum of absolute deviations from the prior matrix \code{m}, subject to directional net flow constraints:
\deqn{\sum_j x_{ji} - \sum_j x_{ij} = \text{net}_i}
Structural zeros are enforced by the \code{zero_mask}. All flows are constrained to be non-negative.
}
\examples{
m <- matrix(c(0, 100, 30, 70,
              50,   0, 45,  5,
              60,  35,  0, 40,
              20,  25, 20,  0),
            nrow = 4, byrow = TRUE,
            dimnames = list(orig = LETTERS[1:4], dest = LETTERS[1:4]))
addmargins(m)
sum_region(m)

net <- c(30, 40, -15, -55)
result <- net_matrix_lp(net_tot = net, m = m)
result$n |>
  addmargins() |>
  round(2)
sum_region(result$n)
}
\seealso{
\code{\link{net_matrix_entropy}()} for KL divergence minimization,
\code{\link{net_matrix_ipf}()} for iterative proportional fitting (IPF),
and \code{\link{net_matrix_optim}()} for least-squares (L2) flow estimation.
}
