% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nls_tac.R
\name{nls_tac}
\alias{nls_tac}
\title{Nonlinear fit with the TAC algorithm}
\usage{
nls_tac(
  formula,
  data = parent.frame(),
  functions = NULL,
  nlparam,
  lp_bounds = NULL,
  N = 10,
  tol = 1e-04,
  parallel = FALSE,
  maxiter = 50,
  quiet = FALSE,
  silent = TRUE,
  compute_errors = TRUE
)
}
\arguments{
\item{formula}{A formula given in the form "LHS ~ a1 * F_1(x,p1) + a2 * F_2(x,p2) + ... + an F_n(x,pn)"}

\item{data}{Data frame with the data points to be fitted.}

\item{functions}{A string array with the nonlinear functions. If \code{get_functions} fails to properly provide the functions they should be explicitly introduced.}

\item{nlparam}{A list with the names of the nonlinear parameters and their lower and upper bounds in the form \code{c(lower,upper)}.}

\item{lp_bounds}{An optional list with the bounding restrictions over the linear parameters.}

\item{N}{Size of the partition of the nonlinear parameters. Defaults to 10.}

\item{tol}{Stopping condition. The algorithm stops whenever the maximum difference between two consecutive iterations is less than \code{tol}. Default value is 1e-4}

\item{parallel}{Logical. If TRUE then multicore parallelization of for loops is done with the parallel package. Defaults to FALSE.}

\item{maxiter}{Integer. The maximum number of iterations. Defaults to 50.}

\item{quiet}{Logical. If TRUE, all progress messages are supressed (defaults to FALSE).}

\item{silent}{Logical. Parameter to be passed to get_best_parameters function. If TRUE (default) suppresses any warnings regarding the collinearity of the columns of the matrix in the determination of the best linear parameters.}

\item{compute_errors}{Logical. If TRUE (default value) the function computes the standard error of the estimates.}
}
\value{
An object of class \code{nlstac}.  A list of

\item{\code{coefficients}}{Best coefficients obtained.}

\item{\code{stdError}}{Standard errors for the obtained coefficients}

\item{\code{convInfo}}{Convergence information: a list with the number of iterations performed (\code{niter}) and the tolerance attained at convergence (\code{tol})}

\item{\code{SSR}}{Sum of the squares of the residuals}

\item{\code{resid}}{Residuals}

\item{\code{data}}{Data frame used. Columns of variables not used in the formula fitted will be removed}

\item{\code{formula}}{Formula used}

\item{\code{df}}{Degrees of freedom}

\item{\code{sigma}}{Standard deviation estimate.}

\item{\code{Rmat}}{R matrix in the QR decomposition of the gradient matrix used for the computation of the standard errors of the coefficients}
}
\description{
Fits a nonlinear function to data.
}
\examples{
### Examples from 'nls' doc ###

DNase1 <- subset(DNase, Run == 1)
## using logistic formula
fm2DNase1 <- nls_tac(density ~ Asym/(1 + exp((xmid - log(conc))/scal)),
                   data = DNase1,
                   nlparam = list(xmid = c(1e-7,10), scal = c(1e-7,3)))
## some generics are applicable
coefficients(fm2DNase1)
summary(fm2DNase1)
## obtaining extra information
fm2DNase1$resid # residuals
fm2DNase1$formula # formula used
fm2DNase1$df # degrees of freedom
fm2DNase1$convInfo # Convergence information (n. iterations, tolerance attained)
fm2DNase1$SSR # SSR
fm2DNase1$data$density - fm2DNase1$resid # fitted values

## Synthetic examples

## Double exponential
x <- seq(from = 0, to = 20, length.out = 1000)
y <- 3*exp(-0.12*x) + 0.6*exp(-3.05*x) +  5 + 0.1*rnorm(length(x))
df <- data.frame(time = x, Temp = y)
# The nonlinear parameter list (with lower and upper values)
nlparam <- list(b1 = c(0,2), b2 = c(0,8))
fittac <- nls_tac('Temp ~ a1*exp(-b1*time) +  a2*exp(-b2*time) + a3',
                   data = df,
                   nlparam = nlparam,
                   N = 5)
summary(fittac)
plot(Temp ~ time, data = df)
lines(x, predict(fittac), col = "red", lwd = 2)

##
N <- 100
x <- seq(from = 0, to = 3, length.out = N)
y <- 3*sin(5*x)^2 + 2 + 0.2*rnorm(N)
df <- data.frame(x = x, y = y)
form <- y ~ a1*sin(b1*x)^2 + a2
nlbnds <- list(b1 = c(0.5,10)) # rough bouds for tac
tac_model <-  nls_tac(formula = form,
                      data = df,
                      nlparam = nlbnds,
                      N = 10,
                      tol = 1e-5)
yhat <- predict(tac_model)
plot(x,y)
lines(x,yhat, col = "blue")

}
\references{
Fernández Torvisco, J. A.; Rodríguez-Arias Fernández, M.; Cabello Sánchez, J. (2018). “A New Algorithm to Fit Exponential Decays without Initial Guess”, Filomat 32:12, 4233–4248.

Bates, D. M. and Watts, D. G. (1988) Nonlinear Regression Analysis and Its Applications, Wiley
}
\author{
\strong{Mariano Rodríguez-Arias} (\email{arias@unex.es}).
\emph{Deptartment of Mathematics}

\strong{Juan Antonio Fernández Torvisco} (\email{jfernandck@alumnos.unex.es}).
\emph{Department of Mathematics}

University of Extremadura (Spain)

\strong{Rafael Benítez} (\email{rafael.suarez@uv.es}).
\emph{Department of Business Mathematics}

University of Valencia (Spain)
}
