% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nsp_poly_selfnorm.R
\name{nsp_poly_selfnorm}
\alias{nsp_poly_selfnorm}
\title{Self-normalised Narrowest Significance Pursuit algorithm for piecewise-polynomial signals}
\usage{
nsp_poly_selfnorm(
  y,
  M = 1000,
  thresh.val = NULL,
  power = 1/2,
  min.size = 20,
  alpha = 0.1,
  deg = 0,
  eps = 0.03,
  c = exp(1 + 2 * eps),
  overlap = FALSE
)
}
\arguments{
\item{y}{A vector containing the data sequence.}

\item{M}{The minimum number of intervals considered at each recursive stage, unless the number of all intervals is smaller, in which case all intervals
are used.}

\item{thresh.val}{Numerical value of the significance threshold (lambda in the paper); or \code{NULL} if the threshold is to be determined from
the data.}

\item{power}{A parameter for the (rough) estimator of the global sum of squares of z_t; the span of the moving window in that estimator is
\code{min(n, max(round(n^power), min.size))}, where \code{n} is the length of \code{y}.}

\item{min.size}{(See immediately above.)}

\item{alpha}{Desired maximum probability of obtaining an interval that does not contain a change-point (the significance threshold will be
determined as a function of this parameter).}

\item{deg}{The degree of the polynomial pieces in f_t (0 for the piecewise-constant model; 1 for piecewise-linearity, etc.).}

\item{eps}{Parameter of the self-normalisation statistic as described in the paper; use default if unsure how to set.}

\item{c}{Parameter of the self-normalisation statistic as described in the paper; use default if unsure how to set.}

\item{overlap}{If \code{FALSE}, then on discovering a significant interval, the search continues recursively to the left and to the right of that
interval. If \code{TRUE}, then the search continues to the left and to the right of the midpoint of that interval.}
}
\value{
A list with the following components:
\item{intervals}{A data frame containing the estimated intervals of significance: \code{starts} and \code{ends} is where the intervals start and end,
respectively; \code{values} are the values of the deviation measure on each given interval; \code{midpoints} are their midpoints.}
\item{threshold.used}{The threshold value.}
}
\description{
This function runs the Narrowest Significance Pursuit (NSP) algorithm on a data sequence \code{y} believed to follow the model
y_t = f_t + z_t, where f_t is a piecewise polynomial of degree \code{deg}, and z_t is noise. It returns localised regions (intervals) of the
domain, such that each interval must contain a change-point in the parameters of the polynomial f_t
at the global significance level \code{alpha}.
For any interval considered by the algorithm,
significant departure from parameter constancy is achieved if the multiscale
deviation measure (see Details for the literature reference) exceeds a threshold, which is either provided as input
or determined from the data (as a function of \code{alpha}). The function assumes independence, symmetry and finite variance of the
errors z_t, but little else; in particular they do not need to have a constant variance across t.
}
\details{
The NSP algorithm is described in P. Fryzlewicz (2021) "Narrowest Significance Pursuit: inference for multiple change-points in linear
models", preprint.
}
\examples{
set.seed(1)
g <- c(rep(0, 100), rep(10, 100), rep(0, 100))
x.g <- g + stats::rnorm(300) * seq(from = 1, to = 4, length = 300)
nsp_poly_selfnorm(x.g, 100)
}
\seealso{
\code{\link{nsp_poly}}, \code{\link{nsp_poly}}, \code{\link{nsp_poly_ar}}, \code{\link{nsp_tvreg}}, \code{\link{nsp_selfnorm}}
}
\author{
Piotr Fryzlewicz, \email{p.fryzlewicz@lse.ac.uk}
}
