% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sun.R
\name{sunAngle}
\alias{sunAngle}
\title{Solar Angle as Function of Space and Time}
\usage{
sunAngle(t, longitude = 0, latitude = 0, useRefraction = FALSE)
}
\arguments{
\item{t}{time, either a POSIXt object (converted to timezone \code{"UTC"},
if it is not already in that timezone), or a value (character or numeric)
that can be converted to a time with \code{\link[=as.POSIXct]{as.POSIXct()}}, assuming the
timezone to be \code{"UTC"}.}

\item{longitude}{observer longitude in degrees east.}

\item{latitude}{observer latitude in degrees north.}

\item{useRefraction}{boolean, set to \code{TRUE} to apply a correction for
atmospheric refraction.}
}
\value{
A list containing the following:
\itemize{
\item \code{time} the time
\item \code{azimuth}, in degrees eastward of north, from 0 to 360.
\item \code{altitude}, in degrees above the horizon,  ranging from -90 to 90.
\item \code{diameter}, solar diameter, in degrees.
\item \code{distance} to sun, in astronomical units.
\item \code{declination} angle in degrees, computed with \code{\link[=sunDeclinationRightAscension]{sunDeclinationRightAscension()}}.
\item \code{rightAscension} angle in degrees, computed with \code{\link[=sunDeclinationRightAscension]{sunDeclinationRightAscension()}}.
\if{html}{\figure{starCoords.png}{options: width="400"}}
}
}
\description{
This calculates solar angle, based on a NASA-provided Fortran
program, which (according to comments in the code) is in turn
based on "The Astronomical Almanac".
Note that \code{time} may be a single value or a vector
of values; in the latter case, \code{longitude}, \code{latitude} and \code{useRefraction}
are all made to be of the same length as \code{time}, by calling \code{\link[=rep]{rep()}}. This
addresses the case of finding a time-series of angles at a given spatial
location. For other cases of arguments that are not single values,
either call \code{sunAngle()} repeatedly or, if that is too slow,
use \code{expand.grid()} to set up values of uniform length that
are then supplied to \code{sunAngle()}.
}
\examples{

rise <- as.POSIXct("2011-03-03 06:49:00", tz = "UTC") + 4 * 3600
set <- as.POSIXct("2011-03-03 18:04:00", tz = "UTC") + 4 * 3600
mismatch <- function(lonlat) {
    sunAngle(rise, lonlat[1], lonlat[2])$altitude^2 + sunAngle(set, lonlat[1], lonlat[2])$altitude^2
}
result <- optim(c(1, 1), mismatch)
lonHfx <- (-63.55274)
latHfx <- 44.65
dist <- geodDist(result$par[1], result$par[2], lonHfx, latHfx)
cat(sprintf(
    "Infer Halifax latitude \%.2f and longitude \%.2f; distance mismatch \%.0f km",
    result$par[2], result$par[1], dist
))

}
\references{
Regarding \code{declination} and \code{rightAscension}, see
references in the documentation for \code{\link[=sunDeclinationRightAscension]{sunDeclinationRightAscension()}}.
The other items are based on Fortran code retrieved from
the file \code{sunae.f}, downloaded from the ftp site
\code{climate1.gsfc.nasa.gov/wiscombe/Solar_Rad/SunAngles}
on 2009-11-1.  Comments in that code list as references:

Michalsky, J., 1988: The Astronomical Almanac's algorithm for approximate
solar position (1950-2050), Solar Energy 40, 227-235

The Astronomical Almanac, U.S. Gov't Printing Office, Washington, D.C.
(published every year).

The code comments suggest that the appendix in Michalsky (1988) contains
errors, and declares the use of the following formulae in the 1995 version
the Almanac:
\itemize{
\item p. A12: approximation to sunrise/set times
\item p. B61: solar altitude (AKA elevation) and azimuth
\item p. B62: refraction correction
\item p. C24: mean longitude, mean anomaly, ecliptic
longitude, obliquity of ecliptic, right ascension, declination, Earth-Sun
distance, angular diameter of Sun
\item p. L2: Greenwich mean sidereal time (ignoring T^2, T^3 terms)
}

The code lists authors as Dr. Joe Michalsky and Dr. Lee Harrison (State
University of New York), with modifications by Dr. Warren Wiscombe (NASA
Goddard Space Flight Center).
}
\seealso{
The corresponding function for the moon is \code{\link[=moonAngle]{moonAngle()}}.

Other things related to astronomy: 
\code{\link{angle2hms}()},
\code{\link{eclipticalToEquatorial}()},
\code{\link{equatorialToLocalHorizontal}()},
\code{\link{julianCenturyAnomaly}()},
\code{\link{julianDay}()},
\code{\link{moonAngle}()},
\code{\link{siderealTime}()},
\code{\link{sunDeclinationRightAscension}()}
}
\author{
Dan Kelley
}
\concept{things related to astronomy}
