#' Bhapkar's (1979) test for marginal homogeneity
#'
#' Fits the marginal homogeneity model using WLS.
#'
#' See:
#' Bhapkar, V. P. (1966). A Note on the Equivalence of Two Test Criteria
#' for Hypotheses in Categorical Data. Journal of the American Statistical
#' Association, 61(313), pp.228-235.
#' @importFrom stats pchisq
#' @param n matrix containing the table to analyze
#' @return a list containing the chi-square statistic, the df and p-value.
#' @export
#' @examples
#' Bhapkar_marginal_homogeneity(vision_data)
Bhapkar_marginal_homogeneity <- function(n) {
  if (nrow(n) != ncol(n)) {
    stop("Input matrix must be square")
  }
  r = nrow(n)
  N = sum(n)
  p = n / N
  p_plus = rowSums(p)
  plus_p = colSums(p)
  d = p_plus[1:(r - 1)] - plus_p[1:(r - 1)]
  w = matrix(nrow=(r - 1), ncol=(r - 1))
  for (i in 1:r - 1) {
    for (j in 1:r - 1) {
      w[i, j] = -(p[i, j] + p[j, i]) - (p_plus[i] - plus_p[i]) * (p_plus[j] - plus_p[j])
    }
    w[i, i] = p_plus[i] + plus_p[i] - 2.0 * p[i, i] - (p_plus[i] - plus_p[i])^2
  }
  w_inverse <- solve(w)
  chisq <- N * d %*% w_inverse %*% d
  df <- r - 1
  p <- pchisq(chisq, df, lower.tail = FALSE)
  list(chisq=chisq, df=df, p=p)
}
