% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/piv_MCMC.R
\name{piv_MCMC}
\alias{piv_MCMC}
\title{JAGS/Stan Sampling for Gaussian Mixture Models and Clustering via Co-Association Matrix.}
\usage{
piv_MCMC(
  y,
  k,
  nMC,
  priors,
  piv.criterion = c("MUS", "maxsumint", "minsumnoint", "maxsumdiff"),
  clustering = c("diana", "hclust"),
  software = c("rjags", "rstan"),
  burn = 0.5 * nMC,
  chains = 4,
  cores = 1,
  sparsity = FALSE
)
}
\arguments{
\item{y}{\eqn{N}-dimensional vector for univariate data or
\eqn{N \times D} matrix for multivariate data.}

\item{k}{Number of mixture components.}

\item{nMC}{Number of MCMC iterations for the JAGS/Stan function execution.}

\item{priors}{Input prior hyperparameters (see Details for default options).}

\item{piv.criterion}{The pivotal criterion used for identifying one pivot
for each group. Possible choices are: \code{"MUS", "maxsumint", "minsumnoint",
"maxsumdiff"}.
The default method is \code{"maxsumint"} (see the Details and
the vignette).}

\item{clustering}{The algorithm adopted for partitioning the
\eqn{N} observations into \code{k} groups. Possible choices are \code{"diana"} (default) or
\code{"hclust"} for divisive and agglomerative hierarchical clustering, respectively.}

\item{software}{The selected MCMC method to fit the model: \code{"rjags"} for the JAGS method, \code{"rstan"} for the Stan method.
Default is \code{"rjags"}.}

\item{burn}{The burn-in period (only if method \code{"rjags"} is selected). Default is \code{0.5}\eqn{\times} \code{nMC}.}

\item{chains}{A positive integer specifying the number of Markov chains. The default is 4.}

\item{cores}{The number of cores to use when executing the Markov chains in parallel (only if
\code{software="rstan"}). Default is 1.}

\item{sparsity}{Allows for sparse finite mixtures, default is \code{FALSE}.}
}
\value{
The function gives the MCMC output, the clustering
solutions and the pivotal indexes. Here there is a complete list of outputs.

\item{\code{true.iter}}{ The number of MCMC iterations for which
the number of JAGS/Stan groups exactly coincides with the prespecified
number of groups \code{k}.}
\item{\code{Mu}}{An estimate of the groups' means.}
\item{\code{groupPost}}{ \eqn{true.iter \times N} matrix
with values from \code{1:k} indicating the post-processed group allocation
vector.}
\item{\code{mcmc_mean}}{  If \code{y} is a vector, a \eqn{true.iter \times k}
matrix with the post-processed MCMC chains for the mean parameters; if
\code{y} is a matrix, a \eqn{true.iter \times D \times k} array with
the post-processed MCMC chains for the mean parameters.}
\item{\code{mcmc_sd}}{  If \code{y} is a vector, a \eqn{true.iter \times k}
matrix with the post-processed MCMC chains for the sd parameters; if
\code{y} is a matrix, a \eqn{true.iter \times D} array with
the post-processed MCMC chains for the sd parameters.}
\item{\code{mcmc_weight}}{A \eqn{true.iter \times k}
matrix with the post-processed MCMC chains for the weights parameters.}
\item{\code{mcmc_mean_raw}}{ If \code{y} is a vector, a \eqn{(nMC-burn) \times k} matrix
with the raw MCMC chains for the mean parameters as given by JAGS; if
\code{y} is a matrix, a \eqn{(nMC-burn) \times D \times k} array with the raw MCMC chains
for the mean parameters as given by JAGS/Stan.}
\item{\code{mcmc_sd_raw}}{ If \code{y} is a vector, a \eqn{(nMC-burn) \times k} matrix
with the raw MCMC chains for the sd parameters as given by JAGS/Stan; if
\code{y} is a matrix, a \eqn{(nMC-burn) \times D} array with the raw MCMC chains
for the sd parameters as given by JAGS/Stan.}
\item{\code{mcmc_weight_raw}}{A \eqn{(nMC-burn) \times k} matrix
with the raw MCMC chains for the weights parameters as given by JAGS/Stan.}
\item{\code{C}}{The \eqn{N \times N} co-association matrix constructed from the MCMC sample.}
\item{\code{grr}}{The vector of cluster membership returned by
\code{"diana"} or \code{"hclust"}.}
\item{\code{pivots}}{The vector of indices of pivotal units identified by the selected pivotal criterion.}
\item{\code{model}}{The JAGS/Stan model code. Apply the \code{"cat"} function for a nice visualization of the code.}
\item{\code{stanfit}}{An object of S4 class \code{stanfit} for the fitted model (only if \code{software="rstan"}).}
}
\description{
Perform MCMC JAGS sampling or HMC Stan sampling for Gaussian mixture models, post-process the chains and apply a clustering technique to the MCMC sample. Pivotal units for each group are selected among four alternative criteria.
}
\details{
The function fits univariate and multivariate Bayesian Gaussian mixture models of the form
(here for univariate only):
\deqn{(Y_i|Z_i=j) \sim \mathcal{N}(\mu_j,\sigma_j),}
where the \eqn{Z_i}, \eqn{i=1,\ldots,N}, are i.i.d. random variables, \eqn{j=1,\dots,k},
\eqn{\sigma_j} is the group variance,  \eqn{Z_i \in {1,\ldots,k }} are the
latent group allocation, and
\deqn{P(Z_i=j)=\eta_j.}
The likelihood of the model is then
\deqn{L(y;\mu,\eta,\sigma) = \prod_{i=1}^N \sum_{j=1}^k \eta_j \mathcal{N}(\mu_j,\sigma_j),}
where \eqn{(\mu, \sigma)=(\mu_{1},\dots,\mu_{k},\sigma_{1},\ldots,\sigma_{k})}
are the component-specific parameters and \eqn{\eta=(\eta_{1},\dots,\eta_{k})}
the mixture weights. Let \eqn{\nu} denote a permutation of \eqn{{ 1,\ldots,k }},
and let \eqn{\nu(\mu)= (\mu_{\nu(1)},\ldots,} \eqn{ \mu_{\nu(k)})},
\eqn{\nu(\sigma)= (\sigma_{\nu(1)},\ldots,} \eqn{ \sigma_{\nu(k)})},
\eqn{ \nu(\eta)=(\eta_{\nu(1)},\ldots,\eta_{\nu(k)})} be the
corresponding permutations of \eqn{\mu}, \eqn{\sigma} and \eqn{\eta}.
 Denote by \eqn{V} the set of all the permutations of the indexes
 \eqn{{1,\ldots,k }}, the likelihood above is invariant under any
 permutation \eqn{\nu \in V}, that is
\deqn{
L(y;\mu,\eta,\sigma) = L(y;\nu(\mu),\nu(\eta),\nu(\sigma)).}
As a consequence, the model is unidentified with respect to an
arbitrary permutation of the labels.
When Bayesian inference for the model is performed,
if the prior distribution \eqn{p_0(\mu,\eta,\sigma)} is invariant under a permutation of the indices, then so is the posterior. That is, if \eqn{p_0(\mu,\eta,\sigma) = p_0(\nu(\mu),\nu(\eta),\sigma)}, then
\deqn{
p(\mu,\eta,\sigma| y) \propto p_0(\mu,\eta,\sigma)L(y;\mu,\eta,\sigma)}
is multimodal with (at least) \eqn{k!} modes.

Depending on the selected software, the model parametrization
changes in terms of the prior choices.
Precisely, the JAGS philosophy with the underlying Gibbs sampling
is to use noninformative priors, and conjugate priors are
preferred for computational speed.
Conversely, Stan adopts weakly informative priors,
with no need to explicitly use the conjugacy.
For univariate mixtures, when
\code{software="rjags"} the specification is the same as the function \code{BMMmodel} of the
\code{bayesmix} package:

 \deqn{\mu_j \sim \mathcal{N}(\mu_0, 1/B0inv)}
 \deqn{\sigma_j \sim \mbox{invGamma}(nu0Half, nu0S0Half)}
 \deqn{\eta \sim \mbox{Dirichlet}(1,\ldots,1)}
 \deqn{S0 \sim \mbox{Gamma}(g0Half, g0G0Half),}

 with default values: \eqn{\mu_0=0, B0inv=0.1, nu0Half =10, S0=2,
 nu0S0Half= nu0Half\times S0,
 g0Half = 5e-17, g0G0Half = 5e-33}, in accordance with the default
 specification:

 \code{priors=list(kind = "independence", parameter = "priorsFish",
 hierarchical = "tau")}

 (see \code{bayesmix} for further details and choices).

 When \code{software="rstan"}, the prior specification is:

 \deqn{\mu_j \sim \mathcal{N}(\mu_0, 1/B0inv)}
 \deqn{\sigma_j \sim \mbox{Lognormal}(\mu_{\sigma}, \tau_{\sigma})}
 \deqn{\eta_j \sim \mbox{Uniform}(0,1),}

 with default values: \eqn{\mu_0=0, B0inv=0.1, \mu_{\sigma}=0, \tau_{\sigma}=2}.
The users may specify new hyperparameter values with the argument:

\code{priors=list(mu_0=1, B0inv=0.2, mu_sigma=3, tau_sigma=5)}

For multivariate mixtures, when \code{software="rjags"} the prior specification is the following:

\deqn{ \bm{\mu}_j  \sim \mathcal{N}_D(\bm{\mu}_0, S2)}
\deqn{ \Sigma^{-1} \sim \mbox{Wishart}(S3, D+1)}
\deqn{\eta \sim \mbox{Dirichlet}(\bm{\alpha}),}

where  \eqn{\bm{\alpha}} is a \eqn{k}-dimensional vector
and \eqn{S_2} and \eqn{S_3}
are positive definite matrices. By default, \eqn{\bm{\mu}_0=\bm{0}},
\eqn{\bm{\alpha}=(1,\ldots,1)} and \eqn{S_2} and \eqn{S_3} are diagonal matrices,
with diagonal elements
equal to 1e+05. The user may specify other values for the hyperparameters
\eqn{\bm{\mu}_0, S_2, S_3} and \eqn{\bm{\alpha}} via \code{priors} argument in such a way:


\code{priors =list(mu_0 = c(1,1), S2 = ..., S3 = ..., alpha = ...)}

with the constraint for \eqn{S2} and \eqn{S3} to be positive definite,
and \eqn{\bm{\alpha}} a vector of dimension \eqn{k} with nonnegative elements.

When \code{software="rstan"}, the prior specification is:

\deqn{ \bm{\mu}_j  \sim \mathcal{N}_D(\bm{\mu}_0, LD*L^{T})}
\deqn{L \sim \mbox{LKJ}(\epsilon)}
\deqn{D^*_j \sim \mbox{HalfCauchy}(0, \sigma_d).}

The covariance matrix is expressed in terms of the LDL decomposition as \eqn{LD*L^{T}},
a variant of the classical Cholesky decomposition, where \eqn{L} is a \eqn{D \times D}
lower unit triangular matrix and \eqn{D*} is a \eqn{D \times D} diagonal matrix.
The Cholesky correlation factor \eqn{L} is assigned a LKJ prior with \eqn{\epsilon} degrees of freedom,  which,
combined with priors on the standard deviations of each component, induces a prior on the covariance matrix;
as \eqn{\epsilon \rightarrow \infty} the magnitude of correlations between components decreases,
whereas \eqn{\epsilon=1} leads to a uniform prior distribution for \eqn{L}.
By default, the hyperparameters are \eqn{\bm{\mu}_0=\bm{0}}, \eqn{\sigma_d=2.5, \epsilon=1}.
The user may propose some different values with the argument:


\code{priors=list(mu_0=c(1,2), sigma_d = 4, epsilon =2)}


If \code{software="rjags"} the function performs JAGS sampling using the \code{bayesmix} package
for univariate Gaussian mixtures, and the \code{runjags}
package for multivariate Gaussian mixtures. If \code{software="rstan"} the function performs
Hamiltonian Monte Carlo (HMC) sampling via the \code{rstan} package (see the vignette and the Stan project
for any help).

After MCMC sampling, this function
clusters the units in \code{k} groups,
calls the \code{piv_sel()} function and yields the
pivots obtained from one among four different
methods (the user may specify one among them via \code{piv.criterion}
argument):
 \code{"maxsumint"}, \code{"minsumnoint"}, \code{"maxsumdiff"}
 and \code{"MUS"} (available only if \code{k <= 4})
(see the vignette for thorough details). Due to computational reasons
clarified in the Details section of the function \code{piv_rel}, the
length of the MCMC chains will be minor or equal than the input
argument \code{nMC}; this length, corresponding to the value
\code{true.iter} returned by the procedure, is the number of
MCMC iterations for which
the number of JAGS/Stan groups exactly coincides with the prespecified
number of groups \code{k}.
}
\examples{

### Bivariate simulation

\dontrun{
N   <- 200
k   <- 4
D   <- 2
nMC <- 1000
M1  <- c(-.5,8)
M2  <- c(25.5,.1)
M3  <- c(49.5,8)
M4  <- c(63.0,.1)
Mu  <- rbind(M1,M2,M3,M4)
Sigma.p1 <- diag(D)
Sigma.p2 <- 20*diag(D)
W <- c(0.2,0.8)
sim <- piv_sim(N = N, k = k, Mu = Mu,
               Sigma.p1 = Sigma.p1,
               Sigma.p2 = Sigma.p2, W = W)

## rjags (default)
res <- piv_MCMC(y = sim$y, k =k, nMC = nMC)

## rstan
res_stan <- piv_MCMC(y = sim$y, k =k, nMC = nMC,
                     software ="rstan")

# changing priors
res2 <- piv_MCMC(y = sim$y,
                 priors = list (
                 mu_0=c(1,1),
                 S2 = matrix(c(0.002,0,0, 0.1),2,2, byrow=TRUE),
                 S3 = matrix(c(0.1,0,0,0.1), 2,2, byrow =TRUE)),
                 k = k, nMC = nMC)
}


### Fishery data (bayesmix package)

\dontrun{
library(bayesmix)
data(fish)
y <- fish[,1]
k <- 5
nMC <- 5000
res <- piv_MCMC(y = y, k = k, nMC = nMC)

# changing priors
res2   <- piv_MCMC(y = y,
                   priors = list(kind = "condconjugate",
                   parameter = "priorsRaftery",
                   hierarchical = "tau"),  k =k, nMC = nMC)
}
}
\references{
Egidi, L., Pappadà, R., Pauli, F. and Torelli, N. (2018). Relabelling in Bayesian Mixture
Models by Pivotal Units. Statistics and Computing, 28(4), 957-969.
}
\author{
Leonardo Egidi \email{legidi@units.it}
}
