#' @title
#' Generator of an approximated separation matrix.
#'
#' @description
#' Creates an object of S4 class `BubleyDyerSeparationGenerator` for the computation of approximated
#' separation matrices, starting from a set of random linear extensions, sampled according to the Bubley-Dyer procedure (see Bubley and Dyer, 1999)
#' Actually, this function does not compute the separation matrices, but just the object that will compute them,
#' by using function `BubleyDyerSeparation`.
#'
#' @param poset Object of S4 class `POSet` representing the poset whose separation matrices are to be computed.
#' Argument `poset` must be created by using any function contained in the package aimed at building object of S4 class `POSet`
#' (e.g. [POSet()], [LinearPOSet()], [ProductPOSet()], ...) .
#'
#' @param seed Positive integer to initialize the random linear extension generation.
#'
#' @param type type of separation to be computed. Possible choices are:
#' "symmetric", "asymmetricLower", "asymmetricUpper", "vertical", "horizontal".
#'
#' @param ... additional types of separations to be computed. Possible choices are:
#' "symmetric", "asymmetricLower", "asymmetricUpper", "vertical", "horizontal".
#'
#' @details The symmetric separation associated to elements \eqn{a} and \eqn{b} in the input poset is the average
#' absolute difference between the positions of \eqn{a} and \eqn{b} observed in the sampled linear extensions (whose elements are arranged in ascending order):
#'
#' \eqn{Sep_{ab}=\frac{1}{n}\sum_{i^1}^{n}|Pos_{l_i}(a)-Pos_{l_i}(b)|},
#'
#' where \eqn{n} is the numbers of sampled linear extensions;
#' \eqn{l_i} represents a sampled linear extension and \eqn{Pos_{l_i}(\cdot)} stands for the position of element \eqn{\cdot}
#' into the sequence of poset elements arranged in increasing order according to \eqn{l_i}.
#'
#' Asymmetric lower and upper separations are defined as:
#' \eqn{Sep_{a < b}=\frac{1}{n}\sum_{i^1}^{n}(Pos_{l_i}(b)-Pos_{l_i}(a))\mathbb{1}(a <_{l_i} b)},
#' \eqn{Sep_{b < a}=\frac{1}{n}\sum_{i^1}^{n}(Pos_{l_i}(a)-Pos_{l_i}(b))\mathbb{1}(b <_{l_i} a)},
#' where \eqn{a\leq_{l_i} b} means that \eqn{a} is lower or equal to \eqn{b} in the linear order defined by linear
#' extension \eqn{l_i} and \eqn{\mathbb{1}} is the indicator function. Note that \eqn{Sep_{ab}=Sep_{a < b}+Sep_{a < b}}.
#'
#' Vertical and horizontal separations (\eqn{vSep} and \eqn{hSep}, respectively) are defined as
#'
#' \eqn{vSep_{ab}=|Sep_{a < b}-Sep_{b < a}|} and #' \eqn{hSep_{ab}=Sep_{ab}-vSep_{ab}|}.
#'
#' For a detailed explanation on why \eqn{vSep} and \eqn{hSep} can be interpreted as vertical and horizontal components
#' of the separation between two poset elements, see Fattore et. al (2024).
#'
#' @return
#' An object of S4 class `BubleyDyerSeparationGenerator`.
#'
#' @references Bubley, R., Dyer, M. (1999). Faster random generation of linear extensions.
#' Discrete Mathematics, 201, 81-88. https://doi.org/10.1016/S0012-365X(98)00333-1
#'
#' Fattore, M., De Capitani, L., Avellone, A., and Suardi, A. (2024).
#' A fuzzy posetic toolbox for multi-criteria evaluation on ordinal data systems.
#' Annals of Operations Research, https://doi.org/10.1007/s10479-024-06352-3.
#'
#'
#' @examples
#' el <- c("a", "b", "c", "d")
#'
#' dom <- matrix(c(
#'   "a", "b",
#'   "c", "b",
#'   "b", "d"
#' ), ncol = 2, byrow = TRUE)
#'
#' pos <- POSet(elements = el, dom = dom)
#'
#' BDgen <- BuildBubleyDyerSeparationGenerator(pos, seed = NULL,
#'               type="symmetric", "asymmetricUpper", "vertical")
#'
#' @name BuildBubleyDyerSeparationGenerator
#' @export BuildBubleyDyerSeparationGenerator
BuildBubleyDyerSeparationGenerator <- function(poset, seed, type, ...)  {
  SeparationTypes <- c("symmetric", "asymmetricLower", "asymmetricUpper", "vertical", "horizontal")
  SeparationTypesC <- c("symmetric", "asymmetricLower", "asymmetricUpper")
  if (!methods::is(poset, "POSet")) {
    stop("poset must be of class POSet")
  }
  if (!is.null(seed) && (seed < 0 || seed != round(seed))) {
    stop("seed must be a positive integer")
  }

  if (!(type %in% SeparationTypes)) {
    stop("type TO_DO")
  }
  parameter_list = list(type)
  for(t in list(...)) {
    if (!(t %in% SeparationTypes)) {
      stop(".... TO_DO")
    }
    parameter_list[[length(parameter_list)+1]] = t
  }

  functions_list <- parameter_list
  if (SeparationTypes[4] %in% parameter_list) {
    functions_list[[length(functions_list)+1]] = SeparationTypes[2]
    functions_list[[length(functions_list)+1]] = SeparationTypes[3]
  }
  if (SeparationTypes[5] %in% parameter_list) {
    functions_list[[length(functions_list)+1]] = SeparationTypes[1]
    functions_list[[length(functions_list)+1]] = SeparationTypes[2]
    functions_list[[length(functions_list)+1]] = SeparationTypes[3]
  }
  functions_list <- intersect(functions_list, SeparationTypesC)

  tryCatch({
    ptr <- .Call("_BuildBubleyDyerSeparationGenerator", poset@ptr, seed, functions_list)
    result <- methods::new("BubleyDyerSeparationGenerator", ptr=ptr, types=parameter_list)
    return (result)
    return (0)
  }, error = function(err) {
    err_split <- strsplit(err[[1]], split = ":")
    stop(err_split[[1]][length(err_split[[1]])])
  }) # END tryCatch
}
