#' Generation of rooted binary trees under the Yule model
#'
#' \code{genYuleTree} - Generates a rooted binary tree in \code{phylo}
#' format with the given number of \code{n} leaves under the Yule model.
#' The Yule process is a simple birth-process in which all species have the same
#' speciation rate.
#'
#' @param n Integer value that specifies the desired number of leaves, i.e.,
#' vertices with in-degree 1 and out-degree 0.\cr
#' Due to the restrictions of the \code{phylo} or \code{multiphylo} format,
#' the number of leaves must be at least 2 since there must be at
#' least one edge.
#'
#' @return \code{genYuleTree} A single tree of class \code{phylo} is returned.
#'
#' @references
#'  - This function uses the \code{rtree} function of the \code{ape} package
#'  (E. Paradis, K. Schliep. “ape 5.0: an environment for modern
#'  phylogenetics and evolutionary analyses in R.”
#'  Bioinformatics, 35, 526-528, 2019). \cr
#'  - G. U. Yule. A mathematical theory of evolution, based on the conclusions of
#'  Dr. J. C. Willis, F. R. S. Philosophical Transactions of the Royal Society
#'  of London. Series B, Containing Papers of a Biological
#'  Character, 213(402-410):21–87, 1925. \cr
#'  - E. F. Harding. The probabilities of rooted tree-shapes generated by random
#'  bifurcation. Advances in Applied Probability, 3(1):44–77, 1971.
#'
#' @export
#' @rdname tmYule
#'
#' @examples
#' genYuleTree(n = 5)
genYuleTree <- function(n){
  if(n < 2 || n%%1!=0){
    stop(paste("A tree must have at least 2 leaves, i.e., n>=2 and n must be",
               "an integer."))
  }
  phy <- ape::rtree(n = n, rooted = TRUE,
                    tip.label = paste("t", sample.int(n,n), sep = ""),
                    br = NULL, equiprob = FALSE)
  return(phy)
}
