% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/KochanekBartels.R
\name{KochanekBartels}
\alias{KochanekBartels}
\title{Kochanek-Bartels quaternions spline}
\usage{
KochanekBartels(
  keyRotors,
  keyTimes = NULL,
  tcb = c(0, 0, 0),
  times,
  n_intertimes,
  endcondition = "natural",
  constantSpeed = FALSE
)
}
\arguments{
\item{keyRotors}{a vector of unit quaternions (rotors) to be interpolated}

\item{keyTimes}{the times corresponding to the key rotors; must be an
increasing vector of the same length a \code{keyRotors} if
\code{endcondition = "natural"} or of length one more than number of key
rotors if \code{endcondition = "closed"}}

\item{tcb}{a vector of three numbers respectively corresponding to tension,
continuity and bias}

\item{times}{the times of interpolation; each time must lie within the range
of the key times; this parameter can be missing if \code{keyTimes} is
\code{NULL} and \code{n_intertimes} is not missing, and it is ignored if
\code{constantSpeed=TRUE}}

\item{n_intertimes}{if given, this argument has precedence over \code{times};
\code{keyTimes} can be \code{NULL} and \code{times} is constructed by
linearly interpolating the key times such that there are
\code{n_intertimes - 1} between two key times (so the times are the key
times if \code{n_intertimes = 1})}

\item{endcondition}{start/end conditions, can be \code{"closed"} or
\code{"natural"}}

\item{constantSpeed}{Boolean, whether to re-parameterize the spline to
have constant speed; in this case, \code{"times"} is ignored and you 
must set the interpolating times with the help of \code{n_intertimes}}
}
\value{
A vector of quaternions having the same length as the \code{times}
  vector.
}
\description{
Constructs a quaternions spline by the Kochanek-Bartels
  algorithm.
}
\examples{
library(qsplines)
# Using a Kochanek-Bartels quaternions spline to construct 
#   a spherical curve interpolating some key points on the 
#     sphere of radius 5
    
# helper function: spherical to Cartesian coordinates
sph2cart <- function(rho, theta, phi){
  return(c(
    rho * cos(theta) * sin(phi),
    rho * sin(theta) * sin(phi),
    rho * cos(phi)
  ))
}

# construction of the key points on the sphere
keyPoints <- matrix(nrow = 0L, ncol = 3L)
theta_ <- seq(0, 2*pi, length.out = 9L)[-1L]
phi <- 1.3
for(theta in theta_){
  keyPoints <- rbind(keyPoints, sph2cart(5, theta, phi))
  phi = pi - phi
}
n_keyPoints <- nrow(keyPoints)

# construction of the key rotors; the first key rotor 
#   is the identity quaternion and rotor i sends the 
#     first key point to the i-th key point
keyRotors <- quaternion(length.out = n_keyPoints)
rotor <- keyRotors[1L] <- H1
for(i in seq_len(n_keyPoints - 1L)){
  keyRotors[i+1L] <- rotor <-
    quaternionFromTo(
      keyPoints[i, ]/5, keyPoints[i+1L, ]/5
    ) * rotor
}

# Kochanek-Bartels quaternions spline
\donttest{rotors <- KochanekBartels(
  keyRotors, n_intertimes = 25L, 
  endcondition = "closed", tcb = c(-1, 5, 0)
)

# construction of the interpolating points on the sphere
points <- matrix(nrow = 0L, ncol = 3L)
keyPoint1 <- rbind(keyPoints[1L, ])
for(i in seq_along(rotors)){
  points <- rbind(points, rotate(keyPoint1, rotors[i]))
}

# visualize the result with the 'rgl' package
library(rgl)
spheres3d(0, 0, 0, radius = 5, color = "lightgreen")
spheres3d(points, radius = 0.2, color = "midnightblue")
spheres3d(keyPoints, radius = 0.25, color = "red")}
}
