% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TwoStg_Mopt.R
\name{TwoStg_Mopt}
\alias{TwoStg_Mopt}
\title{Estimate the Two-stage Mean-Optimal Treatment Regime}
\usage{
TwoStg_Mopt(data, regimeClass.stg1, regimeClass.stg2,
  moPropen1 = "BinaryRandom", moPropen2 = "BinaryRandom", max = TRUE,
  s.tol, cl.setup = 1, p_level = 1, it.num = 10, pop.size = 3000,
  hard_limit = FALSE)
}
\arguments{
\item{data}{a data frame, containing variables in the \code{moPropen} and \code{RegimeClass} and 
a component \code{y} as the response.}

\item{regimeClass.stg1}{a formula or a string specifying the Class of treatment regimes
at stage 1, e.g. \code{a1~x1+x2}}

\item{regimeClass.stg2}{a formula or a string specifying the Class of treatment regimes
at stage 2, e.g. \code{a2~x1+a1+x2}}

\item{moPropen1}{The propensity score model for the probability of receiving 
treatment level 1 at the first stage .
When \code{moPropen1} equals the string "BinaryRandom",  the proportion of observations
 receiving treatment level 1 in the sample at the first stage will be employed
 as a good estimate of the probability for each observation.
 Otherwise, this argument should be a formula/string, based on which this function 
 will fit a logistic regression on the treatment level.  e.g. \code{a1~x1}.}

\item{moPropen2}{The propensity score model for the probability of receiving 
treatment level 1 at the second stage .
When \code{moPropen2} equals the string "BinaryRandom",  the proportion of observations
 receiving treatment level 1 in the sample at the second stage will be employed
 as a good estimate of the probability for each observation.
 Otherwise, this argument should be a formula/string,  based on which this function
  will fit a logistic regression on the treatment level.  e.g. \code{a2~x1+a1+x2}.}

\item{max}{logical. If \code{max=TRUE}, it indicates we wish to maximize the marginal
mean; if \code{max=FALSE}, we wish to minimize the marginal mean. 
The default is \code{TRUE}.}

\item{s.tol}{This is the tolerance level used by \code{genoud}. 
Default is \eqn{10^{-5}} times the difference between
the largest and the smallest value in the observed responses.
This is particularly important when it comes to evaluating \code{it.num}.}

\item{cl.setup}{the number of nodes. >1 indicates choosing parallel computing option in 
\code{rgenoud::genoud}. Default is 1.}

\item{p_level}{choose between 0,1,2,3 to indicate different levels of output
from the genetic function. Specifically, 0 (minimal printing),
  1 (normal), 2 (detailed), and 3 (debug.)}

\item{it.num}{integer > 1. This argument will be used in \code{rgeound::geound} function.
    If there is no improvement in the objective function in this number of generations,
\code{rgenoud::genoud} will think that it has found the optimum.}

\item{pop.size}{an integer with the default set to be 3000. This is the population number for the first generation
in the genetic algorithm (\code{rgenoud::genoud}).}

\item{hard_limit}{logical. When it is true the maximum number of generations
         in  \code{rgeound::geound} cannot exceed 100. Otherwise, in this function, only
\code{it.num} softly controls when \code{genoud} stops. Default is \code{FALSE}.}
}
\value{
This function returns an object with 6 objects. Both \code{coef.1}, \code{coef.2}
and \code{coef.orgn.scale.1}, \code{coef.orgn.scale.2}  were normalized to have unit euclidean norm.
\describe{
 \item{\code{coef.1}, \code{coef.2}}{the set of parameters indexing the estimated 
 mean-optimal treatment regime for 
 standardized covariates.}
 \item{\code{coef.orgn.scale.1}, \code{coef.orgn.scale.2}}{the set of parameter 
 indexing the estimated mean-optimal treatment regime for the original input covariates.}
 \item{\code{hatM}}{the estimated marginal mean when the treatment 
         regime indexed by \code{coef.orgn.scale.1} and \code{coef.orgn.scale.2} 
         is applied on the entire population.
          See the 'details' for connection between \code{coef.orgn.scale.k} and
          \code{coef.k}.}
 \item{\code{call}}{the user's call.}
 \item{\code{moPropen1}, \code{moPropen2}}{the user specified propensity score models
 for the first and the second stage respectively}
 \item{\code{regimeClass.stg1},  \code{regimeClass.stg2}}{the user specified 
 class of treatment regimes for the first and the second stage respectively}
}
}
\description{
This function implements the estimator of 
two-stage mean-optimal treatment regime by inverse probability of weighting 
proposed by Baqun Zhang. As there are more than one stage, the second stage
treatment regime could take into account the evolving status of an individual
after the first stage
and the treatment level received in the first stage. We assume the options at 
the two stages are both binary and take the form:
\deqn{d_1(x_{stage1})=I\left(\beta_{10} +\beta_{11}  x_{11} +...+ \beta_{1k}  x_{1k} > 0\right), 
}{
d1(x_stage1)=I(\beta_10 +\beta_11 * x11 +... + \beta_1k * x1k > 0),
}

\deqn{d_2(x_{stage2})=I\left(\beta_{20} +\beta_{21}  x_{21} +...+ \beta_{2j}  x_{2j} > 0\right)}{
d2(x_stage2)=I(\beta_20 + \beta_21 * x21 +... + \beta_2j * x2j > 0) }
}
\details{
Note that all estimation functions in this package use the same type
of standardization on covariates. Doing so would allow us to provide a bounded 
domain of parameters for searching in the genetic algorithm.
 

For every stage \code{k}, \eqn{k=1,2}, this estimated parameters indexing the two-stage mean-optimal treatment regime
 are returned \emph{in two scales:}
\enumerate{
   \item , the returned \code{coef.k} 
     is the set of parameters that we estimated after standardizing
     every covariate available for decision-making
     at stage \code{k} to be in the interval [0, 1]. To be exact, every covariate is 
   subtracted by the smallest observed value and divided by the difference between 
   the largest and the smallest value.  Next, we carried out the algorithm in Wang 2016 to get the estimated
   regime parameters, \code{coef.k}, based on the standardized data. 
   For the identifiability issue, we force the Euclidean norm of \code{coef.k}
   to be 1.

 \item The difference between \code{coef.k} and \code{coef.orgn.scale.k} is that the latter
   set of parameters correspond to the original covariates,
   so the associated decision rule can be applied directly to novel observations. 
   In other words, let \eqn{\beta} denote the estimated parameter in the original 
   scale, then the estimated treatment regime is:  
     \deqn{ d(x)= I\{\beta_0 + \beta_1 x_1 + ... + \beta_k x_k > 0\},}{ d(x)=
       I{\beta_0 + \beta_1*x_1 + ... + \beta_k*x_k > 0},}
   where the \eqn{\beta} values are returned as \code{coef.orgn.scale.k}, and the
   the vector \eqn{(1, x_1,...,x_k)} corresponds to the specified class of treatment
   regimes in the \code{k}th stage.
}

If, for every input covariate, the smallest observed value is exactly 0 and the range 
(i.e. the largest number minus the smallest number) is exactly 1, then the estimated 
\code{coef.k} and \code{coef.orgn.scale.k} will render identical.
}
\examples{
ilogit <- function(x) exp(x)/(1 + exp(x))
GenerateData.2stg <- function(n){
 x1 <- runif(n)
 p1 <- ilogit(-0.5+x1)
 a1 <- rbinom(n, size=1, prob=p1)
 
 x2 <- runif(n, x1, x1+1)
 p2 <- ilogit(-1 + x2)
 a2 <- rbinom(n, size=1, prob=p2)
 
 mean <- 1+x1+a1*(1-3*(x1-0.2)^2) +x2 + a2*(1-x2-x1)
 y <- mean + (1+a1*(x1-0.5)+0.5*a2*(x2-1))*rnorm(n,0,sd = 1)
 return(data.frame(x1,a1,x2,a2,y))
}

n <- 400
testdata <- GenerateData.2stg(n)
\donttest{
fit <- TwoStg_Mopt(data=testdata, 
                   regimeClass.stg1="a1~x1", regimeClass.stg2="a2~x1+a1+x2",
                   moPropen1="a1~x1", moPropen2="a2~x2",
                   cl.setup=2)
fit

fit2 <- TwoStg_Mopt(data=testdata, 
                   regimeClass.stg1="a1~x1", regimeClass.stg2="a2~a1+x1*x2",
                   moPropen1="a1~x1", moPropen2="a2~x2",
                   cl.setup=2)
fit2

}

\dontshow{
set.seed(11000)
testdata2 <- GenerateData.2stg(30)
fit2 <- TwoStg_Mopt(data=testdata2, 
                   regimeClass.stg1="a1~x1", regimeClass.stg2="a2~x2",
                   moPropen1="a1~x1", moPropen2="a2~x2",
                   cl.setup=1, pop.size=500, it.num=1, s.tol=0.3)
fit2}
}
\references{
\insertRef{zhang2013robust}{quantoptr}
}
\author{
Yu Zhou, \email{zhou0269@umn.edu}
}
