#' Parse Configuration
#'
#' A minimal YAML parser for configuration files.
#'
#' @param x A character vector (or single string) containing YAML content.
#' @return A list representing the configuration.
#' @keywords internal
#' @noRd
#' @author
#' Yaoxiang Li
parse_config <- function(x) {
  result <- list()
  lines <- split_lines(x)
  # Remove full-line comments and blank lines.
  lines <- lines[!grepl("^\\s*#", lines)]
  lines <- lines[!sapply(lines, function(line) trim_space(line) == "")]
  if (length(lines) == 0) return(result)

  pattern <- "^(\\s*)([^:]+):\\s*(.*)$"
  keys <- list()
  for (line in lines) {
    m <- regexec(pattern, line)
    match <- regmatches(line, m)[[1]]
    if (length(match) < 4) next
    indent <- match[2]
    key <- trim_space(match[3])
    value <- match[4]

    # Determine nesting level (assumes 2 spaces per level).
    level <- if (nchar(indent) == 0) 1 else floor(nchar(indent)/2) + 1
    keys[[level]] <- key
    if (length(keys) > level) keys <- keys[1:level]

    parsed_val <- parse_config_value(value)
    result <- set_nested_value(result, keys[1:level], parsed_val)
  }
  result
}

#' Parse Configuration Value
#'
#' Parses a YAML value by stripping inline comments (starting with " #"),
#' removing surrounding quotes, and handling inline arrays.
#'
#' @param x A character string representing a YAML value.
#' @return The parsed value (numeric, logical, vector, or string).
#' @keywords internal
#' @noRd
#' @author
#' Yaoxiang Li
parse_config_value <- function(x) {
  x <- trim_space(x)
  if (x == "") return(list())

  # If the value is quoted, remove the surrounding quotes.
  if (substr(x, 1, 1) %in% c('"', "'")) {
    q <- substr(x, 1, 1)
    closing <- regexpr(q, substr(x, 2, nchar(x)))
    if (closing > 0) {
      closing <- closing + 1
      return(substr(x, 2, closing - 1))
    }
  }

  # Remove inline comments (a space followed by #).
  pos <- regexpr(" #", x)
  if (pos > 0) {
    x <- trim_space(substr(x, 1, pos - 1))
  }

  # Handle inline arrays: e.g., [item1, item2]
  if (grepl("^\\[.*\\]$", x)) {
    inner <- sub("^\\[", "", x)
    inner <- sub("\\]$", "", inner)
    parts <- unlist(strsplit(inner, ","))
    parts <- sapply(parts, trim_space, USE.NAMES = FALSE)
    res <- lapply(parts, parse_config_value)
    flat <- tryCatch(unlist(res, use.names = FALSE), error = function(e) res)
    return(flat)
  }

  # Convert numeric values.
  if (grepl("^[0-9]+(\\.[0-9]+)?$", x)) {
    num <- as.numeric(x)
    if (!is.na(num)) return(num)
  }

  # Convert booleans.
  if (tolower(x) == "true") return(TRUE)
  if (tolower(x) == "false") return(FALSE)

  # Convert "null" to NULL.
  if (tolower(x) == "null") return(NULL)

  # Otherwise, return as string.
  x
}

#' Set Nested Value
#'
#' Recursively sets a value in a nested list given a vector of keys.
#'
#' @param lst The list to update.
#' @param keys A character vector of keys.
#' @param value The value to set.
#' @return The updated list.
#' @keywords internal
#' @noRd
#' @author
#' Yaoxiang Li
set_nested_value <- function(lst, keys, value) {
  if (length(keys) == 1) {
    lst[[keys[[1]]]] <- value
  } else {
    key <- keys[[1]]
    if (is.null(lst[[key]])) {
      lst[[key]] <- list()
    }
    lst[[key]] <- set_nested_value(lst[[key]], keys[-1], value)
  }
  lst
}
