% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mrd_impute.R
\name{mrd_impute}
\alias{mrd_impute}
\title{Multiple Imputation of Multivariate Regression Discontinuity Estimation}
\usage{
mrd_impute(
  formula,
  data,
  subset = NULL,
  cutpoint = NULL,
  bw = NULL,
  front.bw = NA,
  m = 10,
  k = 5,
  kernel = "triangular",
  se.type = "HC1",
  cluster = NULL,
  impute = NULL,
  verbose = FALSE,
  less = FALSE,
  est.cov = FALSE,
  est.itt = FALSE,
  local = 0.15,
  ngrid = 250,
  margin = 0.03,
  boot = NULL,
  method = c("center", "univ", "front"),
  t.design = NULL,
  stop.on.error = TRUE
)
}
\arguments{
\item{formula}{The formula of the MRDD; a symbolic description of the model to be fitted. This is supplied in the
format of \code{y ~ x1 + x2} for a simple sharp MRDD or \code{y ~ x1 + x2 | c1 + c2}
for a sharp MRDD with two covariates. A fuzzy MRDD may be specified as
\code{y ~ x1 + x2 + z} where \code{x1} is the first running variable,
\code{x2} is the second running variable, and 
\code{z} is the endogenous treatment variable. Covariates are then included in the 
same manner as in a sharp MRDD.}

\item{data}{An optional data frame containing the variables in the model. If not found in \code{data},
the variables are taken from \code{environment(formula)}.}

\item{subset}{An optional vector specifying a subset of observations to be used in the fitting process.}

\item{cutpoint}{A numeric vector of length 2 containing the cutpoints at which assignment to the treatment is determined.
The default is \code{c(0, 0)}.}

\item{bw}{A vector specifying the bandwidths at which to estimate the RD. 
Possible values are \code{"IK09"}, \code{"IK12"}, and a user-specified non-negative numeric vector specifying the bandwidths at which to estimate the RD.
The default is \code{"IK12"}. If \code{bw} is \code{"IK12"}, the bandwidth is calculated using the Imbens-Kalyanaraman 
2012 method. If \code{bw}  is \code{"IK09"}, the bandwidth is calculated using 
the Imbens-Kalyanaraman 2009 method. Then the RD is estimated
with that bandwidth, half that bandwidth, and twice that bandwidth. 
If only a single value is passed into the function,
the RD will similarly be estimated at that bandwidth, half that bandwidth, 
and twice that bandwidth.}

\item{front.bw}{A non-negative numeric vector of length 3 specifying the bandwidths at which to estimate the RD for each
of three effects models (complete model, heterogeneous treatment model, and treatment only model) 
detailed in Wong, Steiner, and Cook (2013). If \code{NA}, \code{front.bw} will be determined by cross-validation. The default is \code{NA}.}

\item{m}{A non-negative integer specifying the number of uniformly-at-random samples to draw as search candidates for \code{front.bw},
if \code{front.bw} is \code{NA}. The default is 10.}

\item{k}{A non-negative integer specifying the number of folds for cross-validation to determine \code{front.bw},
if \code{front.bw} is \code{NA}. The default is 5.}

\item{kernel}{A string indicating which kernel to use. Options are \code{"triangular"} 
(default and recommended), \code{"rectangular"}, \code{"epanechnikov"}, \code{"quartic"}, 
\code{"triweight"}, \code{"tricube"}, and \code{"cosine"}.}

\item{se.type}{This specifies the robust standard error calculation method to use,
from the "sandwich" package. Options are,
as in \code{\link[sandwich]{vcovHC}}, \code{"HC3"}, \code{"const"}, \code{"HC"}, \code{"HC0"}, 
\code{"HC1"}, \code{"HC2"}, \code{"HC4"}, \code{"HC4m"}, \code{"HC5"}.
The default is \code{"HC1"}. This option is overridden by \code{cluster}.}

\item{cluster}{An optional vector of length n specifying clusters within which the errors are assumed
to be correlated. This will result in reporting cluster robust SEs. This option overrides
anything specified in \code{se.type}. It is suggested that data with a discrete running 
variable be clustered by each unique value of the running variable (Lee and Card, 2008).}

\item{impute}{An optional vector of length n containing a grouping variable that 
specifies the imputed variables with missing values.}

\item{verbose}{A logical value indicating whether to print additional information to 
the terminal. The default is \code{FALSE}.}

\item{less}{Logical. If \code{TRUE}, return the estimates of parametric linear
and optimal bandwidth non-parametric models only. If \code{FALSE} 
return the estimates of linear, quadratic, and cubic parametric models and 
optimal, half and double bandwidths in non-parametric models. The default is \code{FALSE}.}

\item{est.cov}{Logical. If \code{TRUE}, the estimates of covariates will be included.
If \code{FALSE}, the estimates of covariates will not be included. The default is \code{FALSE}. This option is not
applicable if method is \code{"front"}.}

\item{est.itt}{Logical. If \code{TRUE}, the estimates of intent-to-treat (ITT) will be returned.
If \code{FALSE}, the estimates of ITT will not be returned. The default is \code{FALSE}. This option is not
applicable if method is \code{"front"}.}

\item{local}{A non-negative numeric value specifying the range of neighboring points around the cutoff on the 
standardized scale, for each assignment variable. The default is 0.15.}

\item{ngrid}{A non-negative integer specifying the number of non-zero grid points on each assignment variable,
which is also the number of zero grid points on each assignment variable. The default is 250. The value used in 
Wong, Steiner and Cook (2013) is 2500, which may cause long computational time.}

\item{margin}{A non-negative numeric value specifying the range of grid points beyond the minimum and maximum
of sample points on each assignment variable. The default is 0.03.}

\item{boot}{An optional non-negative integer specifying the number of bootstrap samples to obtain standard error of estimates.
This argument is not optional if method is \code{"front"}.}

\item{method}{A string specifying the method to estimate the RD effect. Options are \code{"center"}, 
\code{"univ"}, \code{"front"}, based on the centering, univariate, and frontier
approaches (respectively) from Wong, Steiner, and Cook (2013).}

\item{t.design}{A character vector of length 2 specifying the treatment option according to design.
The first entry is for \code{x1} and the second entry is for \code{x2}. Options are  
\code{"g"} (treatment is assigned if \code{x1} is greater than its cutoff),
\code{"geq"} (treatment is assigned if \code{x1} is greater than or equal to its cutoff),
\code{"l"} (treatment is assigned if \code{x1} is less than its cutoff),
and \code{"leq"} (treatment is assigned if \code{x1} is less than or equal to its cutoff).
The same options are available for \code{x2}.}

\item{stop.on.error}{A logical value indicating whether to remove bootstraps which cause error in the \code{integrate} function. If \code{TRUE}, bootstraps which cause error are removed
and resampled until the specified number of 
bootstrap samples are acquired. If \code{FALSE}, bootstraps which cause error are not removed. The default is \code{TRUE}.}
}
\value{
\code{mrd_impute} returns an object of \link{class} "\code{mrd}" or \code{"mrdi"} for \code{"front"} method.
  The function \code{summary} is used to obtain and print a summary of the 
  estimated regression discontinuity. The object of class \code{mrd} is a list 
  containing the following components for each estimated treatment effect,
  \code{tau_MRD} or \code{tau_R} and \code{tau_M}:
\item{call}{The matched call.}
\item{type}{A string denoting either \code{"sharp"} or \code{"fuzzy"} RDD.}
\item{cov}{The names of covariates.}
\item{bw}{Numeric vector of each bandwidth used in estimation.}
\item{obs}{Vector of the number of observations within the corresponding bandwidth.}
\item{model}{For a sharp design, a list of the \code{lm} objects is returned.
  For a fuzzy design, a list of lists is returned, each with two elements: 
  \code{firststage}, the first stage \code{lm} object, and \code{iv}, the \code{ivreg} object. 
  A model is returned for each parametric and non-parametric case and corresponding bandwidth.}
\item{frame}{Returns the model frame used in fitting.}
\item{na.action}{The observations removed from fitting due to missingness.}
\item{est}{Numeric vector of the estimate of the discontinuity in the outcome under 
  a sharp MRDD or the Wald estimator in the fuzzy MRDD, for each corresponding bandwidth.}
\item{d}{Numeric vector of the effect size (Cohen's d) for each estimate.}
\item{se}{Numeric vector of the standard error for each corresponding bandwidth.}
\item{z}{Numeric vector of the z statistic for each corresponding bandwidth.}
\item{df}{Numeric vector of the degrees of freedom computed using Barnard and Rubin (1999)
  adjustment for imputation.}
\item{p}{Numeric vector of the p-value for each corresponding bandwidth.}
\item{ci}{The matrix of the 95% confidence interval, \code{c("CI Lower Bound", "CI Upper Bound")} 
  for each corresponding bandwidth.}
\item{impute}{A logical value indicating whether multiple imputation is used or not.}
}
\description{
\code{mrd_impute} estimates treatment effects in a multivariate regression discontinuity design (MRDD) with imputed missing values.
}
\examples{
set.seed(12345)
x1 <- runif(300, -1, 1)
x2 <- runif(300, -1, 1)
cov <- rnorm(300)
y <- 3 + 2 * (x1 >= 0) + 3 * cov + 10 * (x2 >= 0) + rnorm(300)
imp <- rep(1:3, each = 100)
# all examples below have smaller numbers of m to keep run-time low
# centering
mrd_impute(y ~ x1 + x2 | cov, impute = imp, method = "center", t.design = c("geq", "geq"), m = 3)
# univariate
mrd_impute(y ~ x1 + x2 | cov, impute = imp, method = "univ", t.design = c("geq", "geq"), m = 3)
# frontier - don't run due to computation time
\dontrun{mrd_impute(y ~ x1 + x2 | cov, impute = imp, method = "front",
                    boot = 1000, t.design = c("geq", "geq"), m = 3)}
}
\references{
Wong, V. C., Steiner, P. M., Cook, T. D. (2013). 
  Analyzing regression-discontinuity designs with multiple assignment variables: 
  A comparative study of four estimation methods.
  Journal of Educational and Behavioral Statistics, 38(2), 107-141.
  \url{https://journals.sagepub.com/doi/10.3102/1076998611432172}.

Lee, D. S., Lemieux, T. (2010).
  Regression Discontinuity Designs in Economics.
  Journal of Economic Literature, 48(2), 281-355. 
  \doi{10.1257/jel.48.2.281}.

Lee, D. S., Card, D. (2010).
  Regression discontinuity inference with specification error. 
  Journal of Econometrics, 142(2), 655-674. 
  \doi{10.1016/j.jeconom.2007.05.003}.

Barnard, J., Rubin, D. (1999).
  Small-Sample Degrees of Freedom with Multiple Imputation.
  Biometrika, 86(4), 948-55.
}
