% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reReg.R
\name{reReg}
\alias{reReg}
\title{Fits Semiparametric Regression Models for Recurrent Event Data}
\usage{
reReg(
  formula,
  data,
  subset,
  model = "cox",
  B = 0,
  se = c("boot", "sand"),
  control = list()
)
}
\arguments{
\item{formula}{a formula object, with the response on the left of a "~" operator,
and the predictors on the right.
The response must be a recurrent event survival object as returned by function \code{Recur}.}

\item{data}{an optional data frame in which to interpret the variables occurring
in the \code{"formula"}.}

\item{subset}{an optional logical vector specifying a subset of observations to be used
in the fitting process.}

\item{model}{a character string specifying the underlying model.
The available functional form for the rate function and the hazard function include a Cox-type model,
an accelerated mean model, an accelerated rate model, or a generalized scale-change model,
and can be specified via "cox", "am", "ar", or "gsc", respectively.
The rate function and hazard function separated by "\code{|}".
See \bold{Details}.}

\item{B}{a numeric value specifies the number of bootstraps for variance estimation.
When \code{B = 0}, variance estimation will not be performed.}

\item{se}{a character string specifying the method for the variance estimation. See \bold{Details}.
\describe{
   \item{\code{boot}}{ nonparametric bootstrap approach}
   \item{\code{sand}}{ resampling-based sandwich estimator}
}}

\item{control}{a list of control parameters. See \code{\link{reReg.control}} for default values.}
}
\description{
Fits a general (joint) semiparametric regression model for the recurrent event data,
where the rate function of the underlying recurrent event process and
the hazard function of the terminal event can be specified as a Cox-type model,
an accelerated mean model, an accelerated rate model, or a generalized scale-change model.
See details for model specifications.
}
\details{
\bold{Model specification:}

Suppose the recurrent event process and the failure events are
observed in the time interval \eqn{t\in[0,\tau]},
for some constant \eqn{\tau}.
We formulate the recurrent event rate function, \eqn{\lambda(t)},
and the terminal event hazard function, \eqn{h(t)}, 
in the form of
\deqn{\lambda(t) = Z \lambda_0(te^{X^\top\alpha}) e^{X^\top\beta}, h(t) = Z h_0(te^{X^\top\eta})e^{X^\top\theta},}
where \eqn{\lambda_0(t)} is the baseline rate function,
\eqn{h_0(t)} is the baseline hazard function,
\eqn{X} is a \eqn{n} by \eqn{p} covariate matrix and \eqn{\alpha},
\eqn{Z} is an unobserved shared frailty variable, and
\eqn{(\alpha, \eta)} and \eqn{(\beta, \theta)} correspond to the shape and size parameters,
respectively.
The model includes several popular semiparametric models as special cases,
which can be specified via the \code{model} argument with the rate function
and the hazard function separated by "\code{|}".
For examples,
Wang, Qin and Chiang (2001) (\eqn{\alpha = \eta = \theta = 0})
can be called with \code{model = "cox"};
Huang and Wang (2004) (\eqn{\alpha = \eta = 0})
can be called with \code{model = "cox|cox"};
Xu et al. (2017) (\eqn{\alpha = \beta} and \eqn{\eta = \theta})
can be called with \code{model = "am|am"};
Xu et al. (2019) (\eqn{\eta = \theta = 0}) can be called with \code{model = "gsc"}.
Users can mix the models depending on the application. For example,
\code{model = "cox|ar"} postulate a Cox proportional model for the
recurrent event rate function and an accelerated rate model for
the terminal event hazard function (\eqn{\alpha = \theta = 0}).
If only one model is specified without an "\code{|}",
it is used for both the rate function and the hazard function.
For example, specifying \code{model = "cox"} is equivalent to \code{model = "cox|cox"}.
Some models that assumes \code{Z = 1} and requires independent
censoring are also implemented in \code{reReg};
these includes \code{model = "cox.LWYY"} for Lin et al. (2000),
\code{model = "cox.GL"} for Ghosh and Lin (2002),
and \code{model = "am.GL"} for Ghosh and Lin (2003).
Additionally, an improved estimation of the proportional rate model
(Huang and Huang 2022) can be called by \code{model = "cox.HH"} with
additional \code{control} options to specify the underlying procedure.
See \href{https://www.sychiou.com/reReg/articles/reReg-reg.html}{online vignette}
for a detailed discussion of the implemented regression models.

\bold{Variance estimation:}

The available methods for variance estimation are:
\describe{
  \item{boot}{performs nonparametric bootstrap.}
  \item{sand}{performs the efficient resampling-based variance estimation.}
}

\bold{Improving proportional rate model:}
A common semiparametric regression model for recurrent event process
under the noninformative censoring assumption is the Cox-type proportional rate model
(available in \code{reReg()} via \code{model = "cox.LWYY"}).
However, the construction of the pseudo-partial score function ignores the
dependency among recurrent events and thus could be inefficient. 
To improve upon this popular method, Huang and Huang (2022) proposed to combine
a system of weighted pseudo-partial score equations via the generalized method of moments (GMM)
and empirical likelihood (EL) estimation.
The proposed GMM and EL procedures are available in \code{reReg} via \code{model = "cox.HH"}
with additional control specifications.
See \href{https://www.sychiou.com/reReg/articles/reReg-cppl.html}{online vignette}
for an illustration of this feature.

\bold{Control options:}

The \code{control} list consists of the following parameters:
\describe{
  \item{tol}{absolute error tolerance.}
  \item{init}{a list contains initial guesses used for root search.}
  \item{solver}{the equation solver used for root search.
The available options are \code{BB::BBsolve}, \code{BB::dfsane}, \code{BB::BBoptim}, 
\code{optimx::optimr}, \code{dfoptim::hjk}, \code{dfoptim::mads}, \code{optim},
and \code{nleqslv::nleqslv}.}
  \item{eqType}{a character string indicating whether the log-rank type estimating equation or
the Gehan-type estimating equation (when available) will be used. }
  \item{boot.parallel}{an logical value indicating whether parallel computation
will be applied when \code{se = "boot"} is called.}
  \item{boot.parCl}{an integer value specifying the number of CPU cores to be used when
\code{parallel = TRUE}. The default value is half the CPU cores on the current host.}
\item{cppl}{A character string indicating either to improve the proportional rate model via
the generalized method of moments (\code{cppl = "GMM"}) or empirical likelihood estimation (\code{cppl = "EL"}).
This option is only used when \code{model = "cox.HH"}.}
\item{cppl.wfun}{A list of (up to two) weight functions to be combined with the weighted pseudo-partial likelihood scores.
Available options are \code{"Gehan"} and \code{"cumbase"},
which correspond to the Gehan's weight and the cumulative baseline hazard function, respectively.
Alternatively, the weight functions can be specified with function formulas.
This option is only used when \code{model = "cox.HH"}.}
\item{trace}{A logical variable denoting whether some of the
intermediate results of iterations should be displayed to the user.  Default is \code{FALSE}.}
}
}
\examples{
data(simDat)

## Nonparametric estimate
plot(reReg(Recur(t.start \%to\% t.stop, id, event, status) ~ 1, data = simDat, B = 50))

fm <- Recur(t.start \%to\% t.stop, id, event, status) ~ x1 + x2
## Fit the Cox rate model
summary(reReg(fm, data = simDat, model = "cox", B = 50))
## Fit the joint Cox/Cox model
summary(reReg(fm, data = simDat, model = "cox|cox", B = 50))
## Fit the scale-change rate model
summary(reReg(fm, data = simDat, model = "gsc", B = 50, se = "sand"))
}
\references{
Chiou, S.H., Xu, G., Yan, J. and Huang, C.-Y. (2023). Regression Modeling for Recurrent Events Possibly with an Informative Terminal Event Using {R} Package {reReg}.
\emph{Journal of Statistical Software}, \bold{105}(5): 1--34.

Lin, D., Wei, L., Yang, I. and Ying, Z. (2000). Semiparametric Regression for the Mean and Rate Functions of Recurrent Events.
\emph{Journal of the Royal Statistical Society: Series B (Methodological)}, \bold{62}: 711--730.

Wang, M.-C., Qin, J., and Chiang, C.-T. (2001). Analyzing Recurrent Event Data with Informative Censoring.
\emph{Journal of the American Statistical Association}, \bold{96}(455): 1057--1065.

Ghosh, D. and Lin, D.Y. (2002). Marginal Regression Models for Recurrent and Terminal Events. \emph{Statistica Sinica}: 663--688.

Ghosh, D. and Lin, D.Y. (2003). Semiparametric Analysis of Recurrent Events Data in the Presence of Dependent Censoring.
\emph{Biometrics}, \bold{59}: 877--885.

Huang, C.-Y. and Wang, M.-C. (2004). Joint Modeling and Estimation for Recurrent Event Processes and Failure Time Data.
\emph{Journal of the American Statistical Association}, \bold{99}(468): 1153--1165.

Xu, G., Chiou, S.H., Huang, C.-Y., Wang, M.-C. and Yan, J. (2017). Joint Scale-change Models for Recurrent Events and Failure Time.
\emph{Journal of the American Statistical Association}, \bold{112}(518): 796--805.

Xu, G., Chiou, S.H., Yan, J., Marr, K., and Huang, C.-Y. (2019). Generalized Scale-Change Models for Recurrent Event
Processes under Informative Censoring. \emph{Statistica Sinica}, \bold{30}: 1773--1795.

Huang, M.-Y. and Huang, C.-Y. (2022). Improved semiparametric estimation of the proportional rate model with recurrent event data.
\emph{Biometrics}, \bold{79} 3: 1686--1700.
}
\seealso{
\code{\link{Recur}}, \code{\link{simGSC}}
}
