# roxyglobals

<!-- badges: start -->
[![R-CMD-check](https://github.com/anthonynorth/roxyglobals/actions/workflows/R-CMD-check.yaml/badge.svg)](https://github.com/anthonynorth/roxyglobals/actions/workflows/R-CMD-check.yaml)
[![Codecov test coverage](https://codecov.io/gh/anthonynorth/roxyglobals/branch/master/graph/badge.svg)](https://app.codecov.io/gh/anthonynorth/roxyglobals?branch=master)
[![CRAN](https://www.r-pkg.org/badges/version/roxyglobals)](https://CRAN.R-project.org/package=roxyglobals)
[![r-universe](https://anthonynorth.r-universe.dev/badges/roxyglobals)](https://anthonynorth.r-universe.dev/roxyglobals)
[![latest](https://img.shields.io/github/r-package/v/anthonynorth/roxyglobals?label=latest&logo=r)](https://github.com/anthonynorth/roxyglobals)
<!-- badges: end -->

Generate `utils::globalVariables()` from roxygen @autoglobal and @global tags.

## Installation

```r
# Install the released version from CRAN
install.packages("roxyglobals")

# Install the released version from r-universe
install.packages("roxyglobals", repos = "https://anthonynorth.r-universe.dev")

# Or the development version from GitHub:
# install.packages("remotes")
remotes::install_github("anthonynorth/roxyglobals")
```

## Setup

Add roxyglobals to an R package DESCRIPTION via:

```r
# Add to current R package
roxyglobals::use_roxyglobals()

# Or add to another R package
# install.packages("withr")
withr::with_dir("path/to/package", roxyglobals::use_roxyglobals())
```

## Config

By default, roxyglobals writes all discovered globals (including duplicates) to `R/globals.R`. You may configure the filename and global generation behaviour with:

```r
# write globals to R/generated-globals.R
roxyglobals::options_set_filename("generated-globals.R")

# only emit unique globals
roxyglobals::options_set_unique(TRUE)
```

## Usage

Add @autoglobal to a function roxygen comment block, example:

```r
#' Summarise responses
#'
#' @name summarise_responses
#' @param responses a data.frame of responses
#'
#' @autoglobal
#' @export
summarise_responses <- function(responses) {
  # station_name, station_type, end_time, start_time need to be added to 
  # utils::globalVariables() to keep R CMD CHECK happy
  responses |>
    dplyr::group_by(station_name, station_type) |>
    dplyr::summarise(
      count_responses = dplyr::n(),
      total_hours = sum(
        as.numeric(end_time - start_time, units = "hours"),
        na.rm = TRUE
      ),
      .groups = "drop"
    )
}
```

Or @global, example:

```r
#' Summarise responses
#'
#' @name summarise_responses
#' @param responses a data.frame of responses
#'
#' @global station_name station_type end_time start_time
#' @export
summarise_responses <- function(responses) {
  # station_name, station_type, end_time, start_time need to be added to 
  # utils::globalVariables() to keep R CMD CHECK happy
  responses |>
    dplyr::group_by(station_name, station_type) |>
    dplyr::summarise(
      count_responses = dplyr::n(),
      total_hours = sum(
        as.numeric(end_time - start_time, units = "hours"),
        na.rm = TRUE
      ),
      .groups = "drop"
    )
}
```

Run `devtools::document()` to generate `utils::globalVariables()` in your globals file (default `R/globals.R`). Example globals file:

```r
# Generated by roxyglobals: do not edit by hand

utils::globalVariables(c(
  "end_time", # <summarise_responses>
  "start_time", # <summarise_responses>
  "station_name", # <summarise_responses>
  "station_type", # <summarise_responses>
  NULL
))
```
