% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/unequal_probability_sampling.R
\name{Unequal probability sampling}
\alias{Unequal probability sampling}
\alias{rpm}
\alias{spm}
\alias{cps}
\alias{poisson}
\alias{conditional_poisson}
\alias{systematic}
\alias{systematic_random_order}
\alias{brewer}
\alias{pareto}
\alias{sampford}
\title{Unequal probability sampling}
\usage{
rpm(probabilities, ...)

spm(probabilities, ...)

cps(probabilities, ...)

poisson(probabilities, ...)

conditional_poisson(probabilities, sample_size, ...)

systematic(probabilities, ...)

systematic_random_order(probabilities, ...)

brewer(probabilities, ...)

pareto(probabilities, ...)

sampford(probabilities, ...)
}
\arguments{
\item{probabilities}{A vector of inclusion probabilities.}

\item{...}{
  Arguments passed on to \code{\link[=.sampling_defaults]{.sampling_defaults}}
  \describe{
    \item{\code{eps}}{A small value used when comparing floats.}
    \item{\code{max_iter}}{The maximum number of iterations used in iterative algorithms.}
  }}

\item{sample_size}{The wanted sample size}
}
\value{
A vector of sample indices.
}
\description{
Selects samples with prescribed inclusion probabilities from finite populations.
}
\details{
\code{sampford} and \code{conditional_poisson} may return an error if a solution is not found within
\code{max_iter}.
}
\section{Functions}{
\itemize{
\item \code{rpm()}: Random pivotal method

\item \code{spm()}: Sequential pivotal method

\item \code{cps()}: Correlated Poisson sampling

\item \code{poisson()}: Poisson sampling

\item \code{conditional_poisson()}: Conditional Poisson sampling

\item \code{systematic()}: Systematic sampling

\item \code{systematic_random_order()}: Systematic sampling with random order

\item \code{brewer()}: Brewer sampling

\item \code{pareto()}: Pareto sampling

\item \code{sampford()}: Sampford sampling

}}
\examples{
set.seed(12345);
N = 1000;
n = 100;
prob = rep(n/N, N);
xs = matrix(runif(N * 2), ncol = 2);

s = rpm(prob);
plot(xs[, 1], xs[, 2], pch = ifelse(sample_to_indicator(s, N), 19, 1));

s = spm(prob);
plot(xs[, 1], xs[, 2], pch = ifelse(sample_to_indicator(s, N), 19, 1));

s = cps(prob);
plot(xs[, 1], xs[, 2], pch = ifelse(sample_to_indicator(s, N), 19, 1));

s = poisson(prob);
plot(xs[, 1], xs[, 2], pch = ifelse(sample_to_indicator(s, N), 19, 1));

s = brewer(prob);
plot(xs[, 1], xs[, 2], pch = ifelse(sample_to_indicator(s, N), 19, 1));

s = pareto(prob);
plot(xs[, 1], xs[, 2], pch = ifelse(sample_to_indicator(s, N), 19, 1));

s = systematic(prob);
plot(xs[, 1], xs[, 2], pch = ifelse(sample_to_indicator(s, N), 19, 1));

s = systematic_random_order(prob);
plot(xs[, 1], xs[, 2], pch = ifelse(sample_to_indicator(s, N), 19, 1));

# Conditional poisson and sampford are not guaranteed to find a solution
prob2 = rep(0.5, 10L);
s = conditional_poisson(prob2, 5L, max_iter = 10000L);
plot(xs[1:10, 1], xs[1:10, 2], pch = ifelse(sample_to_indicator(s, 10L), 19, 1));

s = sampford(prob2, max_iter = 10000L);
plot(xs[1:10, 1], xs[1:10, 2], pch = ifelse(sample_to_indicator(s, 10L), 19, 1));

\donttest{
# Respects inclusion probabilities
set.seed(12345);
prob = c(0.2, 0.25, 0.35, 0.4, 0.5, 0.5, 0.55, 0.65, 0.7, 0.9);
N = length(prob);

ep = rep(0L, N);
r = 10000L;

for (i in seq_len(r)) {
  s = poisson(prob);
  ep[s] = ep[s] + 1L;
}

print(ep / r - prob);
}

}
\references{
Bondesson, L., & Thorburn, D. (2008).
A list sequential sampling method suitable for real‐time sampling.
Scandinavian Journal of Statistics, 35(3), 466-483.

Brewer, K. E. (1975).
A Simple Procedure for Sampling pi-ps wor.
Australian Journal of Statistics, 17(3), 166-172.

Chauvet, G. (2012).
On a characterization of ordered pivotal sampling.
Bernoulli, 18(4), 1320-1340.

Deville, J.-C., &  Tillé, Y. (1998).
Unequal probability sampling without replacement through a splitting method.
Biometrika 85, 89-101.

Grafström, A. (2009).
Non-rejective implementations of the Sampford sampling design.
Journal of Statistical Planning and Inference, 139(6), 2111-2114.

Rosén, B. (1997).
On sampling with probability proportional to size.
Journal of statistical planning and inference, 62(2), 159-191.

Sampford, M. R. (1967).
On sampling without replacement with unequal probabilities of selection.
Biometrika, 54(3-4), 499-513.
}
