% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run_drt.R
\name{run_drt}
\alias{run_drt}
\alias{run_drt.default}
\alias{run_drt.formula}
\title{Run Doubly Ranked Tests}
\usage{
run_drt(X, G, method = c("suff.rank", "avg.rank"), data.names = NULL)

\method{run_drt}{default}(X, G, method = c("suff.rank", "avg.rank"), data.names = NULL)

\method{run_drt}{formula}(formula, ...)
}
\arguments{
\item{X}{an n by T matrix or an S by T by n array containing the functions (or surfaces) to analyze.}

\item{G}{a vector of length n containing the grouping variable.}

\item{method}{statistic for summarizing the ranks: 'suff.rank' for sufficient statistic (the default) or
'avg.rank' for arithmetic average.}

\item{data.names}{a vector of length two containing names that describe \code{X} and \code{G}.}

\item{formula}{a formula of the form \code{X ~ G}.}

\item{...}{additional arguments to pass to \code{\link[=run_drt.default]{run_drt.default()}}, e.g. \code{method}.}
}
\value{
A list with class "\code{htest}" containing the following components:\tabular{ll}{
\code{statistic} \tab the value of the test statistic with a name describing it. \cr
\tab \cr
\code{parameter} \tab  the parameter(s) for the exact distribution of the test statistic. \cr
\tab \cr
\code{p.value} \tab the p-value for the test. \cr
\tab \cr
\code{null.value} \tab the location parameter. \cr
\tab \cr
\code{alternative} \tab a character string describing the alternative hypothesis. \cr
\tab \cr
\code{data.name} \tab a character string giving the names of the data. \cr
\tab \cr
\code{test_details} \tab the output from the internally run Wilcoxon rank sum or Kruskal-Wallis test. \cr
\tab \cr
\code{method} \tab character string giving the type of doubly ranked test performed. \cr
\tab \cr
\code{ranks} \tab a list containing the ranks by column (if \code{X} is a matrix) and the summarized ranks. \cr
\tab \cr
\code{data} \tab a list containing \code{X} and \code{G}. \cr
}
}
\description{
Performs two (or more) sample doubly ranked tests on pre-processed functional data,
formatted as either a matrix (for functions) or an array (for surfaces).
}
\details{
Doubly ranked tests are non-parametric tests that first rank functions (or surfaces) by time (or location). Next, the procedure
summarizes the observed ranks using a statistic. The summarized ranks are then analyzed using either a Wilcoxon rank sum
test or a Kruskal-Wallis test. To perform a doubly ranked test, realizations of functions must be stored in an n by T matrix where
n is the total number of observed functions and T is the number of realizations per function (commonly time points or locations). Surface
data in an S by T by n array can be analyzed as well, although currently this feature has under gone only limited testing.

By default, \code{run_drt()} implements a sufficient statistic when summarizing the ranks of each observed function across T, i.e.
the argument \code{method} defaults to \code{method = suff.rank}. This statistic has the form
\deqn{t(z) = \frac{1}{T}\sum_{t=1}^T\log\left[ \left(\frac{z_t}{n}- \frac{1}{2n}\right)\bigg/\left(1-\frac{z_t}{n} + \frac{1}{2n}\right) \right],}
where \eqn{z_t} is the observed rank at time \eqn{t}. See Meyer (2024) for additional details. The average rank may also be
used by setting \code{method = 'avg.rank'}, although this summary has not undergone testing in the doubly ranked context.

Regardless of the statistic used, the summarized ranks are the analyzed using either \code{\link[=wilcox.test]{wilcox.test()}} or \code{\link[=kruskal.test]{kruskal.test()}},
depending on the number of groups in \code{G}.

For functional data, Meyer (2024) suggests using \code{\link[refund:fpca.face]{refund::fpca.face()}} for pre-processing the data, but \code{X} can be pre-processed using any functional
data approach or it can just be the raw data. \code{run_drt()} itself performs no pre-processing and takes \code{X} as inputted.
}
\examples{
#### Two Sample Problem: Resin Viscosity ####
library(FDboost)
data("viscosity")

Xv    <- matrix(viscosity$visAll, nrow = nrow(viscosity$visAll), ncol = ncol(viscosity$visAll))
fXv   <- refund::fpca.face(Xv)
Yvis  <- fXv$Yhat
TR  <- viscosity$T_A

run_drt(Yvis ~ TR)

#### Four Sample Problem: Canadian Weather ####
R     <- fda::CanadianWeather$region
XT    <- t(fda::CanadianWeather$dailyAv[,,'Temperature.C'])
fXT   <- refund::fpca.face(XT)
YT    <- fXT$Yhat

run_drt(YT ~ R)

}
\references{
Meyer, MJ (2024). Doubly ranked tests for grouped functional data. \emph{Available on arXiv} at \url{https://arxiv.org/abs/2306.14761}.
}
