% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/index.R
\name{get_index}
\alias{get_index}
\alias{get_index_split}
\alias{get_cog}
\alias{get_weighted_average}
\alias{get_eao}
\title{Extract a relative biomass/abundance index, center of gravity, effective
area occupied, or weighted average}
\usage{
get_index(obj, bias_correct = TRUE, level = 0.95, area = 1, silent = TRUE, ...)

get_index_split(
  obj,
  newdata,
  bias_correct = FALSE,
  nsplit = 1,
  level = 0.95,
  area = 1,
  silent = FALSE,
  predict_args = list(),
  ...
)

get_cog(
  obj,
  bias_correct = FALSE,
  level = 0.95,
  format = c("long", "wide"),
  area = 1,
  silent = TRUE,
  ...
)

get_weighted_average(
  obj,
  vector,
  bias_correct = FALSE,
  level = 0.95,
  area = 1,
  silent = TRUE,
  ...
)

get_eao(obj, bias_correct = FALSE, level = 0.95, area = 1, silent = TRUE, ...)
}
\arguments{
\item{obj}{Output from \code{\link[=predict.sdmTMB]{predict.sdmTMB()}} with \code{return_tmb_object = TRUE}
(the usual case). Alternatively, if \code{\link[=sdmTMB]{sdmTMB()}} was called with \code{do_index = TRUE}, or if using \code{\link[=get_index_split]{get_index_split()}}, an object from \code{\link[=sdmTMB]{sdmTMB()}}.}

\item{bias_correct}{Should bias correction be implemented via
\code{\link[TMB:sdreport]{TMB::sdreport()}}? Bias correction accounts for the non-linear
transformation of random effects when calculating the index. Recommended to
be \code{TRUE} for final analyses, but can be set to \code{FALSE} for faster
calculation while experimenting with models. See Thorson and Kristensen
(2016) in the References.}

\item{level}{The confidence level.}

\item{area}{Grid cell area for area weighting the index. Can be: (1) a
numeric vector of length \code{nrow(newdata)} with area for each grid cell, (2)
a single numeric value to apply to all grid cells, or (3) a character value
giving the column name in \code{newdata} containing areas.}

\item{silent}{Silent?}

\item{...}{Passed to \code{\link[TMB:sdreport]{TMB::sdreport()}}.}

\item{newdata}{New data (e.g., a prediction grid by year) to pass to
\code{\link[=predict.sdmTMB]{predict.sdmTMB()}} in the case of \code{get_index_split()}.}

\item{nsplit}{The number of splits to do the calculation in. For memory
intensive operations (large grids and/or models), it can be helpful to
do the prediction, area integration, and bias correction on subsets of
time slices (e.g., years) instead of all at once. If \code{nsplit > 1}, this
will usually be slower but with reduced memory use.}

\item{predict_args}{A list of arguments to pass to \code{\link[=predict.sdmTMB]{predict.sdmTMB()}} in the
case of \code{get_index_split()}.}

\item{format}{Long or wide.}

\item{vector}{A numeric vector of the same length as the prediction data,
containing the values to be averaged (e.g., depth, temperature).}
}
\value{
For \code{get_index()}:
A data frame with columns for time, estimate (area-weighted total abundance
or biomass), lower and upper confidence intervals, log estimate, and standard
error of the log estimate.

For \code{get_cog()}:
A data frame with columns for time, estimate (center of gravity: the
abundance-weighted mean x and y coordinates), lower and upper confidence
intervals, and standard error of center of gravity coordinates.

For \code{get_eao()}:
A data frame with columns for time, estimate (effective area occupied: the
area required if the population was spread evenly at the arithmetic mean
density), lower and upper confidence intervals, log EAO, and standard error
of the log EAO estimates.

For \code{get_weighted_average()}:
A data frame with columns for time, estimate (weighted average of the
provided vector, weighted by predicted density), lower and upper confidence
intervals, and standard error of the estimates.
}
\description{
Extract a relative biomass/abundance index, center of gravity, effective
area occupied, or weighted average
}
\examples{
\dontshow{if (ggplot2_installed()) withAutoprint(\{ # examplesIf}
\donttest{
library(ggplot2)

# use a small number of knots for this example to make it fast:
mesh <- make_mesh(pcod, c("X", "Y"), n_knots = 60)

# fit a spatiotemporal model:
m <- sdmTMB(
 data = pcod,
 formula = density ~ 0 + as.factor(year),
 time = "year", mesh = mesh, family = tweedie(link = "log")
)

# prepare a prediction grid:
nd <- replicate_df(qcs_grid, "year", unique(pcod$year))

# Note `return_tmb_object = TRUE` and the prediction grid:
predictions <- predict(m, newdata = nd, return_tmb_object = TRUE)

# biomass index:
ind <- get_index(predictions, bias_correct = TRUE)
ind
ggplot(ind, aes(year, est)) + geom_line() +
  geom_ribbon(aes(ymin = lwr, ymax = upr), alpha = 0.4) +
  ylim(0, NA)

# do that in 2 chunks
# only necessary for very large grids to save memory
# will be slower but save memory
# note the first argument is the model fit object:
ind <- get_index_split(m, newdata = nd, nsplit = 2, bias_correct = TRUE)

# center of gravity:
cog <- get_cog(predictions, format = "wide")
cog
ggplot(cog, aes(est_x, est_y, colour = year)) +
  geom_point() +
  geom_linerange(aes(xmin = lwr_x, xmax = upr_x)) +
  geom_linerange(aes(ymin = lwr_y, ymax = upr_y)) +
  scale_colour_viridis_c()

# effective area occupied:
eao <- get_eao(predictions)
eao
ggplot(eao, aes(year, est)) + geom_line() +
  geom_ribbon(aes(ymin = lwr, ymax = upr), alpha = 0.4) +
  ylim(0, NA)

# weighted average (e.g., depth-weighted by biomass):
wa <- get_weighted_average(predictions, vector = nd$depth)
wa
ggplot(wa, aes(year, est)) + geom_line() +
  geom_ribbon(aes(ymin = lwr, ymax = upr), alpha = 0.4)
}
\dontshow{\}) # examplesIf}
}
\references{
Geostatistical model-based indices of abundance
(along with many newer papers):

Shelton, A.O., Thorson, J.T., Ward, E.J., and Feist, B.E. 2014. Spatial
semiparametric models improve estimates of species abundance and
distribution. Canadian Journal of Fisheries and Aquatic Sciences 71(11):
1655--1666. \doi{10.1139/cjfas-2013-0508}

Thorson, J.T., Shelton, A.O., Ward, E.J., and Skaug, H.J. 2015.
Geostatistical delta-generalized linear mixed models improve precision for
estimated abundance indices for West Coast groundfishes. ICES J. Mar. Sci.
72(5): 1297–1310. \doi{10.1093/icesjms/fsu243}

Geostatistical model-based centre of gravity:

Thorson, J.T., Pinsky, M.L., and Ward, E.J. 2016. Model-based inference for
estimating shifts in species distribution, area occupied and centre of
gravity. Methods Ecol Evol 7(8): 990–1002. \doi{10.1111/2041-210X.12567}

Geostatistical model-based effective area occupied:

Thorson, J.T., Rindorf, A., Gao, J., Hanselman, D.H., and Winker, H. 2016.
Density-dependent changes in effective area occupied for
sea-bottom-associated marine fishes. Proceedings of the Royal Society B:
Biological Sciences 283(1840): 20161853.
\doi{10.1098/rspb.2016.1853}

Bias correction:

Thorson, J.T., and Kristensen, K. 2016. Implementing a generic method for
bias correction in statistical models using random effects, with spatial and
population dynamics examples. Fisheries Research 175: 66–74.
\doi{10.1016/j.fishres.2015.11.016}
}
\seealso{
\code{\link[=get_index_sims]{get_index_sims()}}
}
