% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lda-shrink-mean.r
\name{lda_shrink_mean}
\alias{lda_shrink_mean}
\alias{lda_shrink_mean.default}
\alias{lda_shrink_mean.formula}
\alias{predict.lda_shrink_mean}
\title{Shrinkage-mean-based Diagonal Linear Discriminant Analysis (SmDLDA) from
Tong, Chen, and Zhao (2012)}
\usage{
lda_shrink_mean(x, ...)

\method{lda_shrink_mean}{default}(x, y, prior = NULL, ...)

\method{lda_shrink_mean}{formula}(formula, data, prior = NULL, ...)

\method{predict}{lda_shrink_mean}(object, newdata, type = c("class", "prob", "score"), ...)
}
\arguments{
\item{x}{Matrix or data frame containing the training data. The rows are the
sample observations, and the columns are the features. Only complete data are
retained.}

\item{...}{additional arguments (not currently used).}

\item{y}{Vector of class labels for each training observation. Only complete
data are retained.}

\item{prior}{Vector with prior probabilities for each class. If NULL
(default), then equal probabilities are used. See details.}

\item{formula}{A formula of the form \code{groups ~ x1 + x2 + ...} That is,
the response is the grouping factor and the right hand side specifies the
(non-factor) discriminators.}

\item{data}{data frame from which variables specified in \code{formula} are
preferentially to be taken.}

\item{object}{Fitted model object}

\item{newdata}{Matrix or data frame of observations to predict. Each row
corresponds to a new observation.}

\item{type}{Prediction type: either \code{"class"}, \code{"prob"}, or \code{"score"}.}
}
\value{
\code{lda_shrink_mean} object that contains the trained SmDLDA classifier
}
\description{
Given a set of training data, this function builds the Shrinkage-mean-based
Diagonal Linear Discriminant Analysis (SmDLDA) classifier from Tong, Chen,
and Zhao (2012). The SmDLDA classifier incorporates a Lindley-type shrunken
mean estimator into the DLDA classifier from Dudoit et al. (2002). For more
about the DLDA classifier, see \code{\link[=lda_diag]{lda_diag()}}.

The SmDLDA classifier is a modification to LDA, where the off-diagonal
elements of the pooled sample covariance matrix are set to zero.
}
\details{
The DLDA classifier belongs to the family of Naive Bayes classifiers, where
the distributions of each class are assumed to be multivariate normal and to
share a common covariance matrix.

The DLDA classifier is a modification to the well-known LDA classifier, where
the off-diagonal elements of the pooled sample covariance matrix are assumed
to be zero -- the features are assumed to be uncorrelated. Under multivariate
normality, the assumption uncorrelated features is equivalent to the
assumption of independent features. The feature-independence assumption is a
notable attribute of the Naive Bayes classifier family. The benefit of these
classifiers is that they are fast and have much fewer parameters to estimate,
especially when the number of features is quite large.

The matrix of training observations are given in \code{x}. The rows of \code{x}
contain the sample observations, and the columns contain the features for each
training observation.

The vector of class labels given in \code{y} are coerced to a \code{factor}.
The length of \code{y} should match the number of rows in \code{x}.

An error is thrown if a given class has less than 2 observations because the
variance for each feature within a class cannot be estimated with less than 2
observations.

The vector, \code{prior}, contains the \emph{a priori} class membership for
each class. If \code{prior} is NULL (default), the class membership
probabilities are estimated as the sample proportion of observations belonging
to each class. Otherwise, \code{prior} should be a vector with the same length
as the number of classes in \code{y}. The \code{prior} probabilities should be
nonnegative and sum to one.
}
\examples{
library(modeldata)
data(penguins)
pred_rows <- seq(1, 344, by = 20)
penguins <- penguins[, c("species", "body_mass_g", "flipper_length_mm")]
smdlda_out <- lda_shrink_mean(species ~ ., data = penguins[-pred_rows, ])
predicted <- predict(smdlda_out, penguins[pred_rows, -1], type = "class")

smdlda_out2 <- lda_shrink_mean(x = penguins[-pred_rows, -1], y = penguins$species[-pred_rows])
predicted2 <- predict(smdlda_out2, penguins[pred_rows, -1], type = "class")
all.equal(predicted, predicted2)
}
\references{
Tong, T., Chen, L., and Zhao, H. (2012), "Improved Mean
Estimation and Its Application to Diagonal Discriminant Analysis,"
Bioinformatics, 28, 4, 531-537.
\url{https://academic.oup.com/bioinformatics/article/28/4/531/211887}

Dudoit, S., Fridlyand, J., & Speed, T. P. (2002). "Comparison of
Discrimination Methods for the Classification of Tumors Using Gene Expression
Data," Journal of the American Statistical Association, 97, 457, 77-87.
}
