% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/group_pts.R
\name{group_pts}
\alias{group_pts}
\title{Group Points}
\usage{
group_pts(
  DT = NULL,
  threshold = NULL,
  id = NULL,
  coords = NULL,
  timegroup,
  splitBy = NULL
)
}
\arguments{
\item{DT}{input data.table}

\item{threshold}{distance for grouping points, in the units of the
coordinates}

\item{id}{Character string of ID column name}

\item{coords}{Character vector of X coordinate and Y coordinate column names}

\item{timegroup}{timegroup field in the DT within which the grouping will be
calculated}

\item{splitBy}{(optional) character string or vector of grouping column
name(s) upon which the grouping will be calculated}
}
\value{
\code{group_pts} returns the input \code{DT} appended with a
\code{group} column.

This column represents the spatialtemporal group. As with the other
grouping functions,  the actual value of \code{group} is arbitrary and
represents the identity of a given group where 1 or more individuals are
assigned to a group. If the data was reordered, the \code{group} may
change, but the contents of each group would not.

A message is returned when a column named \code{group} already exists in
the input \code{DT}, because it will be overwritten.
}
\description{
\code{group_pts} groups rows into spatial groups. The function accepts a
\code{data.table} with relocation data, individual identifiers and a
threshold argument. The threshold argument is used to specify the criteria
for distance between points which defines a group. Relocation data should be
in two columns representing the X and Y coordinates.
}
\details{
The \code{DT} must be a \code{data.table}. If your data is a
\code{data.frame}, you can convert it by reference using
\code{\link[data.table:setDT]{data.table::setDT}}.

The \code{id}, \code{coords}, \code{timegroup} (and optional \code{splitBy})
arguments expect the names of a column in \code{DT} which correspond to the
individual identifier, X and Y coordinates, timegroup (typically generated by
\code{group_times}) and additional grouping columns.

The \code{threshold} must be provided in the units of the coordinates. The
\code{threshold} must be larger than 0. The coordinates must be planar
coordinates (e.g.: UTM). In the case of UTM, a \code{threshold} = 50 would
indicate a 50m distance threshold.

The \code{timegroup} argument is required to define the temporal groups
within which spatial groups are calculated. The intended framework is to
group rows temporally with \code{\link{group_times}} then spatially with
\code{group_pts} (or \code{\link{group_lines}}, \code{\link{group_polys}}).
If you have already calculated temporal groups without
\code{\link{group_times}}, you can pass this column to the \code{timegroup}
argument. Note that the expectation is that each individual will be observed
only once per timegroup. Caution that accidentally including huge numbers of
rows within timegroups can overload your machine since all pairwise distances
are calculated within each timegroup.

The \code{splitBy} argument offers further control over grouping. If within
your \code{DT}, you have multiple populations, subgroups or other distinct
parts, you can provide the name of the column which identifies them to
\code{splitBy}. The grouping performed by \code{group_pts} will only consider
rows within each \code{splitBy} subgroup.
}
\examples{
# Load data.table
library(data.table)
\dontshow{data.table::setDTthreads(1)}

# Read example data
DT <- fread(system.file("extdata", "DT.csv", package = "spatsoc"))

# Select only individuals A, B, C for this example
DT <- DT[ID \%in\% c('A', 'B', 'C')]

# Cast the character column to POSIXct
DT[, datetime := as.POSIXct(datetime, tz = 'UTC')]

# Temporal grouping
group_times(DT, datetime = 'datetime', threshold = '20 minutes')

# Spatial grouping with timegroup
group_pts(DT, threshold = 5, id = 'ID',
          coords = c('X', 'Y'), timegroup = 'timegroup')

# Spatial grouping with timegroup and splitBy on population
group_pts(DT, threshold = 5, id = 'ID', coords = c('X', 'Y'),
         timegroup = 'timegroup', splitBy = 'population')
}
\seealso{
\code{\link{group_times}}

Other Spatial grouping: 
\code{\link{group_lines}()},
\code{\link{group_polys}()}
}
\concept{Spatial grouping}
