#' This class represents an SQL binary operator.
#'
#' Used to generate SQL expressions involving a binary operator like in
#' "a / 10".
#'
#' @examples
#' # To generate "a / 10":
#' ExprBinOp$new(ExprField$new("a"), "/", ExprValue$new(10))
#'
#' @import R6
#' @include ExprComp.R
#' @export
ExprBinOp <- R6::R6Class("ExprBinOp",
  inherit = ExprComp,
  public = list(

    #' @description
    #' Initializer.
    #' @param lexpr An Expr instance for the left part.
    #' @param op The binary operator, as a string.
    #' @param rexpr An Expr instance for the right part.
    #' @param ... Arguments to pass to parent class.
    #' @return Nothing.
    initialize = function(lexpr, op, rexpr, ...) {
      super$initialize(...)
      chk::chk_is(lexpr, "Expr")
      chk::chk_is(rexpr, "Expr")
      chk::chk_string(op)

      private$op <- op
      private$lexpr <- lexpr
      private$rexpr <- rexpr

      return(invisible(NULL))
    },

    #' @description
    #' Generates the list of tokens representing this statement.
    #' @return A list of Token objects.
    getTokens = function() {
      tokens <- list()
      if (private$paren) {
        tokens <- c(tokens, .lparen)
      }
      tokens <- c(
        tokens,
        private$lexpr$getTokens(),
        make_middle_op(private$op),
        private$rexpr$getTokens()
      )
      if (private$paren) {
        tokens <- c(tokens, .rparen)
      }
      return(tokens)
    }
  ),
  private = list(
    op = NULL,
    lexpr = NULL,
    rexpr = NULL
  )
)
