#' Table field definition.
#'
#' Used to define a field when creating a table.
#'
#' @examples
#' # To generate the definition of a field named "title":
#' ExprFieldDef$new("title", "TEXT", nullable = FALSE)
#'
#' @import R6
#' @include Expr.R
#' @export
ExprFieldDef <- R6::R6Class("ExprFieldDef",
  inherit = Expr,
  public = list(

    #' @description
    #' Initializer.
    #' @param name The field name.
    #' @param type The field's type (integer, date, varchar(...), ...).
    #' @param primary Set to TRUE if the field is a PRIMARY KEY.
    #' @param nullable Set to FALSE of the field does not accept NULL values.
    #' @return Nothing.
    initialize = function(name, type, primary = FALSE, nullable = TRUE) {
      chk::chk_string(name)
      chk::chk_string(type)
      chk::chk_flag(primary)
      chk::chk_flag(nullable)
      private$name <- name
      private$type <- type
      private$primary <- primary
      private$nullable <- nullable
    },

    #' @description
    #' Generates the list of tokens representing this statement.
    #' @return A list of Token objects.
    getTokens = function() {
      tokens <- list(TokenIdentifier$new(private$name), .spc,
                     TokenKeyword$new(private$type))
      if (private$primary)
        tokens <- c(tokens, .spc, .primary, .spc, .key)
      if (! private$nullable)
        tokens <- c(tokens, .spc, .not, .spc, .null)
      return(tokens)
    }

  ),
  private = list(
    name = NULL,
    type = NULL,
    primary = FALSE,
    nullable = TRUE
))
