% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/change_tv.r
\name{change_tv}
\alias{change_tv}
\title{Methods to include time-varying parameters in an SS3 operating model}
\usage{
change_tv(change_tv_list, ctl_file_in = "control.ss_new",
  ctl_file_out = "om.ctl", dat_file_in = "ss3.dat",
  dat_file_out = "ss3.dat")
}
\arguments{
\item{change_tv_list}{*A list of named vectors. Names correspond to parameters
in the operating model that currently do not use environmental deviations and
the vectors correspond to deviations. See the section "Specifying the
\code{change_tv_list}" below for help on specifying this argument.}

\item{ctl_file_in}{A string providing the path to the input SS \code{.ctl} file.}

\item{ctl_file_out}{A string providing the path to the output SS control file. If the value is \code{NULL}, the file will not be written to the disk.}

\item{dat_file_in}{A string providing the path to the input SS \code{.dat} file.}

\item{dat_file_out}{A string providing the path to the output SS \code{.dat} file.}
}
\value{
The function creates modified versions of the  \code{.ctl} and
  \code{.dat} files if ctl_file_out and dat_file_out are not NULL. The function
  also returns a list of the modified \code{.ctl} and \code{.dat} R objects
  invisibly.
}
\description{
\code{change_tv} takes SS3 \code{.ctl}, \code{.par}, and \code{.dat} files
and implements time-varying parameters using environmental variables.
\code{change_tv} is specifically set up to work with an operating model
\code{.ctl} file.
}
\details{
Although there are three ways to implement time-varying parameters within
SS3, \pkg{ss3sim} and \code{change_tv} only use the environmental variable
option. Within SS3, time-varying parameters work on an annual time-step.
Thus, for models with multiple seasons, the time-varying parameters will
remain constant for the entire year.

The \code{ctl_file_in} argument needs to be a \code{.ss_new} file because
the documentation in \code{.ss_new} files are automated and standardized.
This function takes advantage of the standard documentation the
\code{.ss_new} files to determine which lines to manipulate and where to
add code in the \code{.ctl}, \code{.par}, and \code{.dat} files, code that
is necessary to implement time-varying parameters.

\pkg{ss3sim} uses annual recruitment deviations and may not work with a
model that ties recruitment deviations to environmental covariates. If you
need to compare the environment to annual recruitment deviations, the
preferred option is to transform the environmental variable into an age 0
pre-recruit survey. See page 55 of the SS3 version 3.24f manual for more
information.
}
\section{Which arguments to specifiy in case files}{

All function argument descriptions that start with an asterisk (*) will be passed
through the case files to \code{\link{run_ss3sim}}. If one of these arguments
is not specified in a case file, then a value of \code{NULL} will be passed,
which may or may not be an appropriate value. Other arguments will be ignored
if specified.
}

\section{Specifying the \code{change_tv_list}}{

Parameters will change to vary with time according to the vectors of
deviations passed to \code{change_tv_list}. Vectors of deviations, also
referred to as environmental data, must have a length equal to \code{
endyr-startyr+1}, where \code{endyr} and \code{startyr} are specified the
\code{.dat} file. Specify years without deviations as zero.

Parameter names must be unique and match the full parameter name in the
\code{.ctl} file. Names for stock recruit parameters must contain "devs",
"R0", or "steep", and only one stock recruit parameter can be time-varying
per model.

This feature will include an *additive* functional linkage between
environmental data and the parameter where the link parameter is fixed at a
value of one and the par value is specified in the \code{.par} file:
\eqn{par'[y] = par + link * env[y]}.

For catchability (\eqn{q}) the *additive* functional linkage is implemented
on the log scale: \eqn{ln(q'[y]) = ln(q) + link * env[y]}
}

\section{Passing arguments to \code{change_tv} through \code{\link{run_ss3sim}}}{

(1) create a case file with an arbitrary letter
not used elsewhere (anything but D, E, F, or R) and (2) include the line
\code{function_type; change_tv} in your case file. For example, you might
want to use M for natural mortality, S for selectivity, or G for growth.
}

\examples{
\dontrun{
# Create a temporary folder for the output and set the working directory:
temp_path <- file.path(tempdir(), "ss3sim-tv-example")
dir.create(temp_path, showWarnings = FALSE)
wd <- getwd()
setwd(temp_path)
on.exit(setwd(wd), add = TRUE)

d <- system.file("extdata", package = "ss3sim")
om <- file.path(d, "models", "cod-om")
dir.create("cod-om")
file.copy(om, ".", recursive = TRUE)
setwd("cod-om")

change_tv(change_tv_list =
            list("NatM_p_1_Fem_GP_1" = c(rep(0, 20), rep(.1, 80)),
                 "SR_BH_steep"=rnorm(100, 0, 0.05)),
          ctl_file_in  = "codOM.ctl",
          ctl_file_out = "example.ctl",
          dat_file_in  = "codOM.dat",
          dat_file_out = "example.dat")

# Clean up:
unlink("cod-om", recursive = TRUE)
}
}
\seealso{
Other change functions: \code{\link{change_data}},
  \code{\link{change_em_binning}}, \code{\link{change_e}},
  \code{\link{change_f_par}}, \code{\link{change_f}},
  \code{\link{change_o}}, \code{\link{change_retro}}
}
\author{
Kotaro Ono, Carey McGilliard, Kelli Johnson, and Kathryn Doering
}
\concept{change functions}
