% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/extract.R
\name{st_extract}
\alias{st_extract}
\alias{st_extract.stars}
\title{Extract cell values at point locations}
\usage{
st_extract(x, ...)

\method{st_extract}{stars}(
  x,
  at,
  ...,
  bilinear = FALSE,
  time_column = attr(at, "time_column") \%||\% attr(at, "time_col"),
  interpolate_time = bilinear,
  FUN = mean,
  resampling = c("nearest", "bilinear", "cubic", "cubicspline"),
  sfc_attribute = NULL
)
}
\arguments{
\item{x}{object of class \code{stars} or \code{stars_proxy}}

\item{...}{passed on to \link{aggregate.stars} when geometries are not exclusively POINT geometries}

\item{at}{object of class \code{sf} or \code{sfc} with geometries, or two-column matrix with coordinate points in rows, indicating where to extract values of \code{x}, or a \code{stars} object with geometry and temporal dimensions (vector data cube)}

\item{bilinear}{logical; use bilinear interpolation rather than nearest neighbour?}

\item{time_column}{character or integer; name or index of a column with time or date values that will be matched to values of the first temporal dimension (matching classes \code{POSIXct}, \code{POSIXt}, \code{Date}, or \code{CFTime}), in \code{x}, after which this dimension is reduced. This is useful to extract data cube values along a trajectory; see https://github.com/r-spatial/stars/issues/352 .}

\item{interpolate_time}{logical; should time be interpolated? if FALSE, time instances are matched using the coinciding or the last preceding time in the data cube.}

\item{FUN}{function used to aggregate pixel values when geometries of \code{at} intersect with more than one pixel}

\item{resampling}{character; resampling method; for method cubic or cubicspline,
`stars_proxy` objects should be used and GDAL should have version >= 3.10.0}

\item{sfc_attribute}{character; if \code{at} is of class \code{stars} should the aggregation be performed
for the attribute geometry rather than the dimension geometry? If \code{NULL} (default),
the aggregation is performed at the dimension geometries, else the name of the attribute geometry to perform the aggregation on.
If the given attribute geometry does not exist, the aggregation defaults to the dimension geometry.}
}
\value{
if \code{at} is of class \code{matrix}, a matrix with extracted values is returned; 
if \code{at} is of class \code{stars} and a temporal dimension was passed to \code{time_column},
a \code{stars} object with the original \code{at} dimensions
and the extracted values as attributes.
otherwise: if \code{x} has more dimensions than only x and y (raster), an 
object of class \code{stars} with POINT geometries replacing x and y raster
dimensions, if this is not the case, an object of \code{sf} with extracted values.
}
\description{
Extract cell values at point locations
}
\details{
points outside the raster are returned as \code{NA} values. For
large sets of points for which extraction is needed, passing a matrix as
to \code{at} may be much faster than passing an \code{sf} or \code{sfc} object.
}
\examples{
tif = system.file("tif/L7_ETMs.tif", package = "stars")
r = read_stars(tif)
pnt = st_sample(st_as_sfc(st_bbox(r)), 10)
st_extract(r, pnt)
st_extract(r, pnt) \%>\% st_as_sf()
st_extract(r[,,,1], pnt)
st_extract(r, st_coordinates(pnt)) # "at" is a matrix: return a matrix
# Extraction on non-POINT geometries
poly = st_buffer(pnt, 1000)
st_extract(r, poly)

# Extraction with time matching
rdate = c(r, r*2, along = "date")
dates = c(Sys.Date()-1, Sys.Date())
rdate = st_set_dimensions(rdate, "date", values = c(dates))

pntsf = st_sf(date = dates, geometry = pnt)
st_extract(split(rdate, "band"), pntsf) # POINT geometries

polysf = st_buffer(pntsf, 1000)
st_extract(split(rdate, "band"), polysf, time_column = "date") # POLYGON geometries

vdc = st_sf(rdm = rnorm(20), polygons = st_buffer(st_sample(st_bbox(pnt), 20), 500),
			geometry = rep(pnt, 2), date = rep(dates, each = 10)) |> 
	st_as_stars(dims = c("geometry", "date"))

(vdc_new = st_extract(split(rdate, "band"), vdc)) # stars vector data cube
merge(vdc_new, name = "band")

### Extraction applied to the geometries inside the vector data cube (cell values)
(vdc_new2 = st_extract(split(rdate, "band"), vdc,
					   sfc_attribute = "polygons")) # stars vector data cube
merge(vdc_new2, name = "band")
}
