#' @name ah1
#' @aliases ah1
#' @title Average Hazard (one sample)
#' @description Calculates the average hazard with survival weight and the corresponding asymptotic confidence interval for a single sample.
#' @details It also provides the log-transformed t-year event rate and restricted mean survival time (RMST) along with their variances as intermediate products.
#' Note that the confidence intervals for the t-year event rate and RMST are based on log-scale values. As a result, the confidence intervals may differ slightly from those generated by the \pkg{surv2sampleComp} or \pkg{survRM2} packages.
#' @author Hajime Uno, Miki Horiguchi
#' @references
#' Uno H and Horiguchi M. Ratio and difference of average hazard with survival weight: new measures to quantify survival benefit of new therapy. Statistics in Medicine. 2023;1-17. <doi:10.1002/sim.9651>
#' @usage ah1(time, status, tau, conf.int=0.95)
#' @param time A numeric vector of follow-up times for right-censored data.
#' @param status A numeric vector indicating the event status; 1 = event occurred, 0 = right censored.
#' @param tau A scalar specifying the end time point (\code{tau}) for calculating the average hazard. If \code{tau = NULL}, the default is the maximum time point at which the risk set size in both groups is at least 10.
#' @param conf.int A numeric value specifying the confidence level for confidence intervals. The default is \code{0.95}.
#' @return An object of class \code{ah1}, which contains the following components:
#' @return \item{need_stop}{Logical indicator. If \code{TRUE}, specify another \code{tau}. The survival probability at the specified \code{tau} needs to be > 0.}
#' @return \item{tau}{The truncation time point used in the analysis.}
#' @return \item{result1}{Estimated average hazard.}
#' @examples
#' #====================================================================
#' # cm214_pfs: The sample reconstructed data of the CheckMate214 study.
#' # The code below reproduces the results reported by
#' # Uno and Horiguchi (StatMed; 2023) in Table 6.
#' #====================================================================
#' D      = cm214_pfs
#' time   = D$time
#' status = D$status
#' tau    = 21
#'
#' b = ah1(time=time, status=status, tau=tau, conf.int=0.95)
#' print(b)

#'@export
ah1 = function(time, status, tau, conf.int=0.95){

  #====================================
  # Warning 
  #====================================
  if(max(time)<tau){
    print(paste0("max(time): ",max(time)," is smaller than tau."))
    #--- S(tau) will be S(max(time)) in the following calculation
  }
    
  if(min(time)<=0){
    print(paste0("time includes either 0 or negative number. Please check the input data."))
    stop()
  }
  
#--- initialize ---
n     = length(time)
indx  = order(time)
x     = time[indx]
d     = status[indx]

#-- distinct failure time (up to tau)
timegrd = unique(sort(c(0,(x[d==1 & x<=tau]), tau)))
k       = length(timegrd)

#-- working matrix to create counting process, at-risk process, etc.
wkN = t(matrix(x, ncol=length(x), nrow=k, byrow=T))
wkD = t(matrix(d, ncol=length(d), nrow=k, byrow=T))
wkT = matrix(timegrd, ncol=length(timegrd), nrow=n, byrow=T)

#-- Ni(t) = I(Xi<=t, delta_i=1) -- counting process for each
Nt  = (wkN <= wkT & wkD == 1)*1
#-- dNi(t) = I(Xi==t, delta_i=1)
dNt = (wkN == wkT & wkD == 1)*1

#-- Yi(t) = I(Xi>=t)  -- at risk process for each
Yt  = (wkN >= wkT)*1

#--- D(t) = cummulative number of events at time t
#---dD(t) = number of events at time t
Dt  = apply(Nt, 2, sum)
dDt = apply(dNt, 2, sum)

#--- H(t): cumulative hazard function (NA estimator)
Yt_bar = apply(Yt, 2, sum)
tmp = dDt/Yt_bar; tmp[Yt_bar==0]=0; 
Ht  = cumsum(tmp)
dHt = diff(c(0,Ht))

#--- S(t): survival function (KM estimator)
ft = funcKM2w(x, d, tau=tau)
St = ft$surv[ft$t_idx %in% timegrd]

#====================================
# S(tau) needs to be away from 0 (>0)
#====================================
if(sum(St%in%0)>0){
  need_stop = TRUE
}else{
  need_stop = FALSE
}


#--- RMST(t) = RMST function
RMSTt = rep(0,k)
for(j in 2:k){
  	wk_time  = timegrd[1:j]
  	wk_surv  = St[1:(j-1)]
  	RMSTt[j] = diff(wk_time)%*%wk_surv
}

#--- G(t)=Pr(X>=t) --
Gt = apply(Yt, 2, mean)

#--- Mi(t) = Ni(t) - \int Yi(u)dHt(u) -- Martingale for each
#--- dMi(t) = dNi(t) - Yi(t)dHt(t) -- derivative
Mt = dMt = matrix(0, nrow=n, ncol=length(timegrd))
for(i in 1:n){
  	 Mt[i,] =  Nt[i,] - cumsum(Yt[i,] * dHt)
   	dMt[i,] = diff(c(0, Mt[i,]))
}

#-- make sure the sum is zero
chk = apply(Mt,2,sum);chk


#--- variance estimates for W^F and W^R (log-transformed F and RMST)
F_tau     = 1-St[k]
F_tau_vec = rep(F_tau, length(RMSTt))

RMST_tau  = RMSTt[k]
RMST_tau_vec = rep(RMST_tau, length(RMSTt))

tmp = dHt * (((1-F_tau_vec)/F_tau_vec)*((1-F_tau_vec)/F_tau_vec)/Gt)
v_F = sum(tmp[dHt!=0])

tmp = dHt * (((RMST_tau_vec-RMSTt)/RMST_tau_vec)*((RMST_tau_vec-RMSTt)/RMST_tau_vec)/Gt)
v_R = sum(tmp[dHt!=0])

#--- variance estimates for W^Q (log-transformed)
tmp = dHt * ((1/F_tau_vec - RMSTt/RMST_tau_vec)*(1/F_tau_vec - RMSTt/RMST_tau_vec)/Gt)
v_Q = sum(tmp[dHt!=0])

#--- variance estimates for W^U (NOT log-transformed)
tmp = dHt * ((1/RMST_tau_vec - F_tau_vec*RMSTt/RMST_tau_vec^2)^2/Gt)
v_U = sum(tmp[dHt!=0])

#===============
# Output
#===============
z_alpha = qnorm(1-(1-conf.int)/2)

out = c()

#--F(tau)--
m   = F_tau; se = sqrt(v_F/n)
tmp = c(log(m), se, log(m)-z_alpha*se, log(m)+z_alpha*se, m, exp(log(m)-z_alpha*se), exp(log(m)+z_alpha*se)); out=rbind(out,tmp)

#--RMST(tau)--
m   = RMST_tau; se = sqrt(v_R/n)
tmp = c(log(m), se, log(m)-z_alpha*se, log(m)+z_alpha*se, m, exp(log(m)-z_alpha*se), exp(log(m)+z_alpha*se)); out=rbind(out,tmp)

#--AH(tau)--
m   = F_tau / RMST_tau; se = sqrt(v_Q/n)
tmp = c(log(m), se, log(m)-z_alpha*se, log(m)+z_alpha*se, m, exp(log(m)-z_alpha*se), exp(log(m)+z_alpha*se)); out=rbind(out,tmp)

#-----------
rownames(out) = c("F(tau)","RMST(tau)","AH(tau)")
colnames(out) = c("log(Est)", "SE(log(Est))", paste0(c("log(low_","log(upp_"), conf.int*100, ")"), "Est", paste0(c("low_","upp_"), conf.int*100))


#-----------
Z=list()
Z$need_stop   = need_stop
Z$tau         = tau
Z$result1     = out

class(Z) = "ah1"

attr(Z, "v_F") = v_F
attr(Z, "v_R") = v_R
attr(Z, "v_Q") = v_Q
attr(Z, "v_U") = v_U

return(Z)
}
