% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tar_rep.R, R/tar_rep_raw.R
\name{tar_rep}
\alias{tar_rep}
\alias{tar_rep_raw}
\title{Batched replication with dynamic branching.}
\usage{
tar_rep(
  name,
  command,
  batches = 1,
  reps = 1,
  rep_workers = 1,
  tidy_eval = targets::tar_option_get("tidy_eval"),
  packages = targets::tar_option_get("packages"),
  library = targets::tar_option_get("library"),
  format = targets::tar_option_get("format"),
  repository = targets::tar_option_get("repository"),
  iteration = targets::tar_option_get("iteration"),
  error = targets::tar_option_get("error"),
  memory = targets::tar_option_get("memory"),
  garbage_collection = targets::tar_option_get("garbage_collection"),
  deployment = targets::tar_option_get("deployment"),
  priority = targets::tar_option_get("priority"),
  resources = targets::tar_option_get("resources"),
  storage = targets::tar_option_get("storage"),
  retrieval = targets::tar_option_get("retrieval"),
  cue = targets::tar_option_get("cue"),
  description = targets::tar_option_get("description")
)

tar_rep_raw(
  name,
  command,
  batches = 1,
  reps = 1,
  rep_workers = 1,
  tidy_eval = targets::tar_option_get("tidy_eval"),
  packages = targets::tar_option_get("packages"),
  library = targets::tar_option_get("library"),
  format = targets::tar_option_get("format"),
  repository = targets::tar_option_get("repository"),
  iteration = targets::tar_option_get("iteration"),
  error = targets::tar_option_get("error"),
  memory = targets::tar_option_get("memory"),
  garbage_collection = targets::tar_option_get("garbage_collection"),
  deployment = targets::tar_option_get("deployment"),
  priority = targets::tar_option_get("priority"),
  resources = targets::tar_option_get("resources"),
  storage = targets::tar_option_get("storage"),
  retrieval = targets::tar_option_get("retrieval"),
  cue = targets::tar_option_get("cue"),
  description = targets::tar_option_get("description")
)
}
\arguments{
\item{name}{Name of the target.
\code{\link[=tar_rep]{tar_rep()}} expects unevaluated \code{name} and \code{command} arguments
(e.g. \code{tar_rep(name = sim, command = simulate())})
whereas \code{\link[=tar_rep_raw]{tar_rep_raw()}} expects an evaluated string for \code{name}
and an evaluated expression object for \code{command}
(e.g. \code{tar_rep_raw(name = "sim", command = quote(simulate()))}).}

\item{command}{R code to run multiple times. Must return a list or
data frame because \code{tar_rep()} will try to append new elements/columns
\code{tar_batch} and \code{tar_rep} to the output to denote the batch
and rep-within-batch IDs, respectively.

\code{\link[=tar_rep]{tar_rep()}} expects unevaluated \code{name} and \code{command} arguments
(e.g. \code{tar_rep(name = sim, command = simulate())})
whereas \code{\link[=tar_rep_raw]{tar_rep_raw()}} expects an evaluated string for \code{name}
and an evaluated expression object for \code{command}
(e.g. \code{tar_rep_raw(name = "sim", command = quote(simulate()))}).}

\item{batches}{Number of batches. This is also the number of dynamic
branches created during \code{tar_make()}.}

\item{reps}{Number of replications in each batch. The total number
of replications is \code{batches * reps}.}

\item{rep_workers}{Positive integer of length 1, number of local R
processes to use to run reps within batches in parallel. If 1,
then reps are run sequentially within each batch. If greater than 1,
then reps within batch are run in parallel using a PSOCK cluster.}

\item{tidy_eval}{Whether to invoke tidy evaluation
(e.g. the \verb{!!} operator from \code{rlang}) as soon as the target is defined
(before \code{tar_make()}). Applies to the \code{command} argument.}

\item{packages}{Character vector of packages to load right before
the target runs or the output data is reloaded for
downstream targets. Use \code{tar_option_set()} to set packages
globally for all subsequent targets you define.}

\item{library}{Character vector of library paths to try
when loading \code{packages}.}

\item{format}{Optional storage format for the target's return value.
With the exception of \code{format = "file"}, each target
gets a file in \verb{_targets/objects}, and each format is a different
way to save and load this file. See the "Storage formats" section
for a detailed list of possible data storage formats.}

\item{repository}{Character of length 1, remote repository for target
storage. Choices:
\itemize{
\item \code{"local"}: file system of the local machine.
\item \code{"aws"}: Amazon Web Services (AWS) S3 bucket. Can be configured
with a non-AWS S3 bucket using the \code{endpoint} argument of
\code{\link[targets:tar_resources_aws]{tar_resources_aws()}}, but versioning capabilities may be lost
in doing so.
See the cloud storage section of
\url{https://books.ropensci.org/targets/data.html}
for details for instructions.
\item \code{"gcp"}: Google Cloud Platform storage bucket.
See the cloud storage section of
\url{https://books.ropensci.org/targets/data.html}
for details for instructions.
\item A character string from \code{\link[targets:tar_repository_cas]{tar_repository_cas()}} for content-addressable
storage.
}

Note: if \code{repository} is not \code{"local"} and \code{format} is \code{"file"}
then the target should create a single output file.
That output file is uploaded to the cloud and tracked for changes
where it exists in the cloud. As of \code{targets} version 1.11.0 and higher,
the local file is no longer deleted after the target runs.}

\item{iteration}{Character of length 1, name of the iteration mode
of the target. Choices:
\itemize{
\item \code{"vector"}: branching happens with \code{vectors::vec_slice()} and
aggregation happens with \code{vctrs::vec_c()}.
\item \code{"list"}, branching happens with \verb{[[]]} and aggregation happens with
\code{list()}. In the case of list iteration, \code{tar_read(your_target)}
will return a list of lists, where the outer list has one element per
batch and each inner list has one element per rep within batch.
To un-batch this nested list, call
\code{tar_read(your_target, recursive = FALSE)}.
\item \code{"group"}: \code{dplyr::group_by()}-like functionality to branch over
subsets of a data frame. The target's return value must be a data
frame with a special \code{tar_group} column of consecutive integers
from 1 through the number of groups. Each integer designates a group,
and a branch is created for each collection of rows in a group.
See the \code{tar_group()} function in \code{targets} to see how you can
create the special \code{tar_group} column with \code{dplyr::group_by()}.
}}

\item{error}{Character of length 1, what to do if the target
stops and throws an error. Options:
\itemize{
\item \code{"stop"}: the whole pipeline stops and throws an error.
\item \code{"continue"}: the whole pipeline keeps going.
\item \code{"null"}: The errored target continues and returns \code{NULL}.
The data hash is deliberately wrong so the target is not
up to date for the next run of the pipeline. In addition,
as of \code{targets} version 1.8.0.9011, a value of \code{NULL} is given
to upstream dependencies with \code{error = "null"} if loading fails.
\item \code{"abridge"}: any currently running targets keep running,
but no new targets launch after that.
\item \code{"trim"}: all currently running targets stay running. A queued
target is allowed to start if:
\enumerate{
\item It is not downstream of the error, and
\item It is not a sibling branch from the same \code{\link[targets:tar_target]{tar_target()}} call
(if the error happened in a dynamic branch).
}

The idea is to avoid starting any new work that the immediate error
impacts. \code{error = "trim"} is just like \code{error = "abridge"},
but it allows potentially healthy regions of the dependency graph
to begin running.
(Visit \url{https://books.ropensci.org/targets/debugging.html}
to learn how to debug targets using saved workspaces.)
}}

\item{memory}{Character of length 1, memory strategy. Possible values:
\itemize{
\item \code{"auto"} (default): equivalent to \code{memory = "transient"} in almost
all cases. But to avoid superfluous reads from disk,
\code{memory = "auto"} is equivalent to \code{memory = "persistent"} for
for non-dynamically-branched targets that other targets
dynamically branch over. For example: if your pipeline has
\code{tar_target(name = y, command = x, pattern = map(x))},
then \code{tar_target(name = x, command = f(), memory = "auto")}
will use persistent memory for \code{x}
in order to avoid rereading all of \code{x}
for every branch of \code{y}.
\item \code{"transient"}: the target gets unloaded
after every new target completes.
Either way, the target gets automatically loaded into memory
whenever another target needs the value.
\item \code{"persistent"}: the target stays in memory
until the end of the pipeline (unless \code{storage} is \code{"worker"},
in which case \code{targets} unloads the value from memory
right after storing it in order to avoid sending
copious data over a network).
}

For cloud-based file targets
(e.g. \code{format = "file"} with \code{repository = "aws"}),
the \code{memory} option applies to the
temporary local copy of the file:
\code{"persistent"} means it remains until the end of the pipeline
and is then deleted,
and \code{"transient"} means it gets deleted as soon as possible.
The former conserves bandwidth,
and the latter conserves local storage.}

\item{garbage_collection}{Logical: \code{TRUE} to run \code{base::gc()}
just before the target runs, in whatever R process it is about to run
(which could be a parallel worker).
\code{FALSE} to omit garbage collection.
Numeric values get converted to \code{FALSE}.
The \code{garbage_collection} option in \code{\link[targets:tar_option_set]{tar_option_set()}}
is independent of the
argument of the same name in \code{\link[targets:tar_target]{tar_target()}}.}

\item{deployment}{Character of length 1. If \code{deployment} is
\code{"main"}, then the target will run on the central controlling R process.
Otherwise, if \code{deployment} is \code{"worker"} and you set up the pipeline
with distributed/parallel computing, then
the target runs on a parallel worker. For more on distributed/parallel
computing in \code{targets}, please visit
\url{https://books.ropensci.org/targets/crew.html}.}

\item{priority}{Deprecated on 2025-04-08 (\code{targets} version 1.10.1.9013).
\code{targets} has moved to a more efficient scheduling algorithm
(\url{https://github.com/ropensci/targets/issues/1458})
which cannot support priorities.
The \code{priority} argument of \code{\link[targets:tar_target]{tar_target()}} no longer has a reliable
effect on execution order.}

\item{resources}{Object returned by \code{tar_resources()}
with optional settings for high-performance computing
functionality, alternative data storage formats,
and other optional capabilities of \code{targets}.
See \code{tar_resources()} for details.}

\item{storage}{Character string to control when the output of the target
is saved to storage. Only relevant when using \code{targets}
with parallel workers (\url{https://books.ropensci.org/targets/crew.html}).
Must be one of the following values:
\itemize{
\item \code{"worker"} (default): the worker saves/uploads the value.
\item \code{"main"}: the target's return value is sent back to the
host machine and saved/uploaded locally.
\item \code{"none"}: \code{targets} makes no attempt to save the result
of the target to storage in the location where \code{targets}
expects it to be. Saving to storage is the responsibility
of the user. Use with caution.
}}

\item{retrieval}{Character string to control when the current target
loads its dependencies into memory before running.
(Here, a "dependency" is another target upstream that the current one
depends on.) Only relevant when using \code{targets}
with parallel workers (\url{https://books.ropensci.org/targets/crew.html}).
Must be one of the following values:
\itemize{
\item \code{"auto"} (default): equivalent to \code{retrieval = "worker"} in almost all
cases. But to avoid unnecessary reads from disk, \code{retrieval = "auto"}
is equivalent to \code{retrieval = "main"} for dynamic branches that
branch over non-dynamic targets. For example: if your pipeline has
\code{tar_target(x, command = f())}, then
\code{tar_target(y, command = x, pattern = map(x), retrieval = "auto")}
will use \code{"main"} retrieval in order to avoid rereading all of \code{x}
for every branch of \code{y}.
\item \code{"worker"}: the worker loads the target's dependencies.
\item \code{"main"}: the target's dependencies are loaded on the host machine
and sent to the worker before the target runs.
\item \code{"none"}: \code{targets} makes no attempt to load its
dependencies. With \code{retrieval = "none"}, loading dependencies
is the responsibility of the user. Use with caution.
}}

\item{cue}{An optional object from \code{tar_cue()} to customize the
rules that decide whether the target is up to date.}

\item{description}{Character of length 1, a custom free-form human-readable
text description of the target. Descriptions appear as target labels
in functions like \code{\link[targets:tar_manifest]{tar_manifest()}} and \code{\link[targets:tar_visnetwork]{tar_visnetwork()}},
and they let you select subsets of targets for the \code{names} argument of
functions like \code{\link[targets:tar_make]{tar_make()}}. For example,
\code{tar_manifest(names = tar_described_as(starts_with("survival model")))}
lists all the targets whose descriptions start with the character
string \code{"survival model"}.}
}
\value{
A list of two targets, one upstream and one downstream.
The upstream target returns a numeric index of batch ids,
and the downstream one dynamically maps over the batch ids
to run the command multiple times.
If the command returns a list or data frame, then
the targets from \code{tar_rep()} will try to append new elements/columns
\code{tar_batch}, \code{tar_rep}, and \code{tar_seed} to the output
to denote the batch, rep-within-batch ID, and random number
generator seed, respectively.

\code{tar_read(your_target)} (on the downstream target with the actual work)
will return a list of lists, where the outer list has one element per
batch and each inner list has one element per rep within batch.
To un-batch this nested list, call
\code{tar_read(your_target, recursive = FALSE)}.
}
\description{
Batching is important for optimizing the efficiency
of heavily dynamically-branched workflows:
\url{https://books.ropensci.org/targets/dynamic.html#batching}.
\code{\link[=tar_rep]{tar_rep()}} replicates a command in strategically sized batches.

\code{\link[=tar_rep]{tar_rep()}} expects unevaluated \code{name} and \code{command} arguments
(e.g. \code{tar_rep(name = sim, command = simulate())})
whereas \code{\link[=tar_rep_raw]{tar_rep_raw()}} expects an evaluated string for \code{name}
and an evaluated expression object for \code{command}
(e.g. \code{tar_rep_raw(name = "sim", command = quote(simulate()))}).
}
\details{
\code{tar_rep()} and \code{tar_rep_raw()} each create two targets:
an upstream local stem
with an integer vector of batch ids, and a downstream pattern
that maps over the batch ids. (Thus, each batch is a branch.)
Each batch/branch replicates the command a certain number of times.
If the command returns a list or data frame, then
the targets from \code{tar_rep()} will try to append new elements/columns
\code{tar_batch}, \code{tar_rep}, and \code{tar_seed} to the output
to denote the batch, rep-within-batch index, and rep-specific seed,
respectively.

Both batches and reps within each batch
are aggregated according to the method you specify
in the \code{iteration} argument. If \code{"list"}, reps and batches
are aggregated with \code{list()}. If \code{"vector"},
then \code{vctrs::vec_c()}. If \code{"group"}, then \code{vctrs::vec_rbind()}.
}
\section{Replicate-specific seeds}{

In ordinary pipelines, each target has its own unique deterministic
pseudo-random number generator seed derived from its target name.
In batched replicate, however, each batch is a target with multiple
replicate within that batch. That is why \code{\link[=tar_rep]{tar_rep()}}
and friends give each \emph{replicate} its own unique seed.
Each replicate-specific seed is created
based on the dynamic parent target name,
\code{tar_option_get("seed")} (for \code{targets} version 0.13.5.9000 and above),
batch index, and rep-within-batch index.
The seed is set just before the replicate runs.
Replicate-specific seeds are invariant to batching structure.
In other words,
\code{tar_rep(name = x, command = rnorm(1), batches = 100, reps = 1, ...)}
produces the same numerical output as
\code{tar_rep(name = x, command = rnorm(1), batches = 10, reps = 10, ...)}
(but with different batch names).
Other target factories with this seed scheme are \code{\link[=tar_rep2]{tar_rep2()}},
\code{\link[=tar_map_rep]{tar_map_rep()}}, \code{\link[=tar_map2_count]{tar_map2_count()}}, \code{\link[=tar_map2_size]{tar_map2_size()}},
and \code{\link[=tar_render_rep]{tar_render_rep()}}.
For the \verb{tar_map2_*()} functions,
it is possible to manually supply your own seeds
through the \code{command1} argument and then invoke them in your
custom code for \code{command2} (\code{set.seed()}, \code{withr::with_seed},
or \code{withr::local_seed()}). For \code{\link[=tar_render_rep]{tar_render_rep()}},
custom seeds can be supplied to the \code{params} argument
and then invoked in the individual R Markdown reports.
Likewise with \code{\link[=tar_quarto_rep]{tar_quarto_rep()}} and the \code{execute_params} argument.
}

\section{Target objects}{

Most \code{tarchetypes} functions are target factories,
which means they return target objects
or lists of target objects.
Target objects represent skippable steps of the analysis pipeline
as described at \url{https://books.ropensci.org/targets/}.
Please read the walkthrough at
\url{https://books.ropensci.org/targets/walkthrough.html}
to understand the role of target objects in analysis pipelines.

For developers,
\url{https://wlandau.github.io/targetopia/contributing.html#target-factories}
explains target factories (functions like this one which generate targets)
and the design specification at
\url{https://books.ropensci.org/targets-design/}
details the structure and composition of target objects.
}

\examples{
if (identical(Sys.getenv("TAR_LONG_EXAMPLES"), "true")) {
targets::tar_dir({ # tar_dir() runs code from a temporary directory.
targets::tar_script({
  list(
    tarchetypes::tar_rep(
      x,
      data.frame(x = sample.int(1e4, 2)),
      batches = 2,
      reps = 3
    )
  )
})
targets::tar_make()
targets::tar_read(x)
targets::tar_script({
  list(
    tarchetypes::tar_rep_raw(
      "x",
      quote(data.frame(x = sample.int(1e4, 2))),
      batches = 2,
      reps = 3
    )
  )
})
targets::tar_make()
targets::tar_read(x)
})
}
}
\seealso{
Other branching: 
\code{\link{tar_map2}()},
\code{\link{tar_map2_count}()},
\code{\link{tar_map2_size}()},
\code{\link{tar_map_rep}()},
\code{\link{tar_rep2}()},
\code{\link{tar_rep_map}()},
\code{\link{tar_rep_map_raw}()}
}
\concept{branching}
