% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/distributions.R
\name{tfd_generalized_pareto}
\alias{tfd_generalized_pareto}
\title{The Generalized Pareto distribution.}
\usage{
tfd_generalized_pareto(
  loc,
  scale,
  concentration,
  validate_args = FALSE,
  allow_nan_stats = TRUE,
  name = NULL
)
}
\arguments{
\item{loc}{The location / shift of the distribution. GeneralizedPareto is a
location-scale distribution. This parameter lower bounds the
distribution's support. Must broadcast with \code{scale}, \code{concentration}.
Floating point \code{Tensor}.}

\item{scale}{The scale of the distribution. GeneralizedPareto is a
location-scale distribution, so doubling the \code{scale} doubles a sample
and halves the density. Strictly positive floating point \code{Tensor}. Must
broadcast with \code{loc}, \code{concentration}.}

\item{concentration}{The shape parameter of the distribution. The larger the
magnitude, the more the distribution concentrates near \code{loc} (for
\code{concentration >= 0}) or near \code{loc - (scale/concentration)} (for
\code{concentration < 0}). Floating point \code{Tensor}.}

\item{validate_args}{Logical, default FALSE. When TRUE distribution parameters are checked
for validity despite possibly degrading runtime performance. When FALSE invalid inputs may
silently render incorrect outputs. Default value: FALSE.}

\item{allow_nan_stats}{Logical, default TRUE. When TRUE, statistics (e.g., mean, mode, variance)
use the value NaN to indicate the result is undefined. When FALSE, an exception is raised if
one or more of the statistic's batch members are undefined.}

\item{name}{name prefixed to Ops created by this class.}
}
\value{
a distribution instance.
}
\description{
The Generalized Pareto distributions are a family of continuous distributions
on the reals. Special cases include \code{Exponential} (when \code{loc = 0},
\code{concentration = 0}), \code{Pareto} (when \code{concentration > 0},
\code{loc = scale / concentration}), and \code{Uniform} (when \code{concentration = -1}).
}
\details{
This distribution is often used to model the tails of other distributions.
As a member of the location-scale family,
\code{X ~ GeneralizedPareto(loc=loc, scale=scale, concentration=conc)} maps to
\code{Y ~ GeneralizedPareto(loc=0, scale=1, concentration=conc)} via
\code{Y = (X - loc) / scale}.

For positive concentrations, the distribution is equivalent to a hierarchical
Exponential-Gamma model with \code{X|rate ~ Exponential(rate)} and
\code{rate ~ Gamma(concentration=1 / concentration, scale=scale / concentration)}.
In the following, \code{samps1} and \code{samps2} are identically distributed:

\if{html}{\out{<div class="sourceCode">}}\preformatted{genp <- tfd_generalized_pareto(loc = 0, scale = scale, concentration = conc)
samps1 <- genp \%>\% tfd_sample(1000)
jd <- tfd_joint_distribution_named(
  list(
    rate =  tfd_gamma(1 / genp$concentration, genp$scale / genp$concentration),
    x = function(rate) tfd_exponential(rate)))
samps2 <- jd \%>\% tfd_sample(1000) \%>\% .$x
}\if{html}{\out{</div>}}

The support of the distribution is always lower bounded by \code{loc}. When
\code{concentration < 0}, the support is also upper bounded by
\code{loc + scale / abs(concentration)}.

Mathematical Details

The probability density function (pdf) is,

\if{html}{\out{<div class="sourceCode">}}\preformatted{pdf(x; mu, sigma, shp, x > mu) =   (1 + shp * (x - mu) / sigma)**(-1 / shp - 1) / sigma
}\if{html}{\out{</div>}}

where:
\itemize{
\item \code{concentration = shp}, any real value,
\item \code{scale = sigma}, \code{sigma > 0},
\item \code{loc = mu}.
}

The cumulative density function (cdf) is,

\if{html}{\out{<div class="sourceCode">}}\preformatted{cdf(x; mu, sigma, shp, x > mu) = 1 - (1 + shp * (x - mu) / sigma)**(-1 / shp)
}\if{html}{\out{</div>}}

Distribution parameters are automatically broadcast in all functions; see
examples for details.
Samples of this distribution are reparameterized (pathwise differentiable).
}
\seealso{
For usage examples see e.g. \code{\link[=tfd_sample]{tfd_sample()}}, \code{\link[=tfd_log_prob]{tfd_log_prob()}}, \code{\link[=tfd_mean]{tfd_mean()}}.
}
