#' @title Plot DBE vs Carbon Atoms
#' @name uplot_dbe_vs_c
#' @family plots
#'
#' @description
#' Creates a scatter plot of DBE (double bond equivalents) vs. number of carbon
#' atoms. Points are color-coded by a selected variable (`z_var`). The plot
#' follows the same stylistic conventions as the other uplot_* functions,
#' including the unified theme and optional UME caption.
#'
#' @inheritParams main_docu
#' @inheritDotParams f_colorz
#'
#' @param df A data.table containing columns:
#'        `12C`, `dbe`, and the variable given in `z_var`.
#' @param z_var Variable used for color coding (default `"norm_int"`).
#' @param plotly If TRUE, returns a plotly interactive plot.
#'
#' @return A ggplot2 object or a plotly object (if `plotly = TRUE`).
#'
#' @examples
#' uplot_dbe_vs_c(mf_data_demo, z_var = "norm_int")
#'
#' @export
uplot_dbe_vs_c <- function(
    df,
    z_var = "norm_int",
    palname = "redblue",
    col_bar = TRUE,
    tf = FALSE,
    logo = TRUE,
    gg_size = 12,
    plotly = FALSE,
    ...
) {

  # --- Column validation -------------------------------------------------------
  required <- c("12C", "dbe", z_var)
  missing  <- setdiff(required, names(df))
  if (length(missing))
    stop("Missing required columns: ", paste(missing, collapse = ", "))

  df <- data.table::as.data.table(df)
  zz <- NULL

  # --- Prepare data -----------------------------------------------------------
  df_plot <- df[, .(`12C`, dbe, zz = get(z_var))]

  # Apply optional log-transform
  if (isTRUE(tf)) {
    if (any(df_plot$zz <= 0))
      stop("Log transform requires positive values.")
    df_plot[, zz := log10(zz)]
  }

  # --- Build color palette -----------------------------------------------------
  pal <- f_colorz(df_plot$zz, palname = palname,
                  return_palette = TRUE, ...)

  # --- Base ggplot -------------------------------------------------------------
  p <- ggplot2::ggplot(df_plot, aes(x = `12C`, y = dbe, color = zz)) +
    ggplot2::geom_point(size = 2, alpha = 0.85) +
    ggplot2::scale_color_gradientn(
      colours = pal,
      name = z_var,
      guide = if (col_bar) "colourbar" else "none"
    ) +
    ggplot2::labs(
      title = "DBE vs Carbon Atoms",
      x = "Carbon atoms",
      y = "DBE",
      caption = if (logo) "UltraMassExplorer" else NULL
    ) +
    theme_uplots(base_size = gg_size)

  # --- Return plotly version if requested -------------------------------------
  if (isTRUE(plotly)) {
    pp <- plotly::ggplotly(p)

    # Ensure caption appears in plotly
    if (logo) {
      pp <- pp |>
        plotly::layout(
          annotations = list(
            list(
              text = "<i>UltraMassExplorer</i>",
              xref = "paper", yref = "paper",
              x = 1, y = -0.12,
              xanchor = "right", yanchor = "auto",
              showarrow = FALSE,
              font = list(size = gg_size, color = "gray40")
            )
          )
        )
    }

    return(pp)
  }

  return(p)
}
