#' @title Add columns containing text data to mastodon activity network dataframes
#'
#' @param net A named list of dataframes \code{nodes} and \code{edges} generated by \code{Create}.
#' @param data A dataframe generated by \code{Collect}.
#' @param ... Additional parameters passed to function. Not used in this method.
#' @param writeToFile Logical. Write data to file. Default is \code{FALSE}.
#' @param verbose Logical. Output additional information. Default is \code{TRUE}.
#' 
#' @return Network as a named list of two dataframes containing \code{$nodes} and \code{$edges} including columns
#'   containing text data.
#'
#' @examples
#' \dontrun{
#' # add text to an activity network
#' net_activity <- collect_mdn |>
#'   Create("activity") |>
#'   AddText(collect_mdn)
#'
#' # network
#' net_activity$nodes
#' net_activity$edges
#' }
#'
#' @aliases AddText.activity.mastodon
#' @name AddText.activity.mastodon
#' @export
AddText.activity.mastodon <- function(net, data, ..., writeToFile = FALSE, verbose = TRUE) {
    if ("tag" %in% class(net)) stop("Network subtype not supported by AddText.", call. = FALSE)
    
    net$nodes <- net$nodes |>
      dplyr::left_join(data$posts |> dplyr::select(.data$id, .data$content.text), by = c("post.id" = "id")) |>
      dplyr::rename("vosonTxt_post" = "content.text")
    
    class(net) <- union(class(net), c("voson.text"))
    
    if (writeToFile) write_output_file(net, "rds", "MastodonActivityNet_Txt", verbose = verbose)
    
    net
  }

#' @title Add columns containing text data to mastodon actor network dataframes
#'
#' @param net A named list of dataframes \code{nodes} and \code{edges} generated by \code{Create}.
#' @param data A dataframe generated by \code{Collect}.
#' @param ... Additional parameters passed to function. Not used in this method.
#' @param writeToFile Logical. Write data to file. Default is \code{FALSE}.
#' @param verbose Logical. Output additional information. Default is \code{TRUE}.
#'
#' @return Network as a named list of two dataframes containing \code{$nodes} and \code{$edges} including columns
#'   containing text data.
#'
#' @examples
#' \dontrun{
#' # add text to an actor network ignoring references to actors at the beginning of
#' # comment text
#' net_actor <- collect_mdn |>
#'   Create("actor") |>
#'   AddText(collect_mdn)
#'
#' # network
#' net_actor$nodes
#' net_actor$edges
#' }
#'
#' @aliases AddText.actor.mastodon
#' @name AddText.actor.mastodon
#' @export
AddText.actor.mastodon <- function(net, data, ..., writeToFile = FALSE, verbose = TRUE) {
    if ("server" %in% class(net)) stop("Network subtype not supported by AddText.", call. = FALSE)
    
    net$edges <- net$edges |>
      dplyr::left_join(data$posts |> dplyr::select(.data$id, .data$content.text), by = c("post.id" = "id")) |>
      dplyr::rename("vosonTxt_post" = "content.text")
    
    class(net) <- union(class(net), c("voson.text"))
    
    if (writeToFile) write_output_file(net, "rds", "MastodonActorNet_Txt", verbose = verbose)
    
    net
  }
